#
#

"""
Display, edit and apply Job Codes.
"""

import logging

from PyQt5.QtCore import QRegularExpression, Qt, pyqtSlot
from PyQt5.QtGui import (
    QFont,
    QIcon,
    QPalette,
    QRegularExpressionValidator,
)
from PyQt5.QtWidgets import (
    QAbstractItemView,
    QCheckBox,
    QComboBox,
    QDialog,
    QDialogButtonBox,
    QGridLayout,
    QHBoxLayout,
    QLabel,
    QListWidgetItem,
    QMessageBox,
    QSizePolicy,
    QVBoxLayout,
    QWidget,
)

from svg.constants import JobCodeSort
from svg.internationalisation.install import install_gettext
from svg.prefs.preferences import Preferences
from svg.tools.utilities import data_file_path
from svg.ui.chevroncombo import ChevronCombo
from svg.ui.messagewidget import MessageButton, MessageWidget
from svg.ui.panelview import QPanelView
from svg.ui.viewutils import (
    FlexiFrame,
    QNarrowListWidget,
    ScrollAreaNoFrame,
    standardIconSize,
    standardMessageBox,
    translateDialogBoxButtons,
)

install_gettext()


class JobCodeDialog(QDialog):
    def __init__(self, parent, on_download: bool, job_codes: list[str]) -> None:
        """
        Prompt user to enter a Job Code, either at the time a download starts,
        or to zero or more selected files before the download begins.

        :param parent: sdrApp main window
        :param on_download: if True, dialog is being prompted for before a download
         starts.
        :param job_codes:
        """

        super().__init__(parent)
        self.sdrApp = parent
        self.prefs: Preferences = self.sdrApp.prefs
        thumbnailModel = self.sdrApp.thumbnailModel

        # Whether the user has opened this dialog before a download starts without
        # having selected any files first
        no_selection_made: bool | None = None

        if on_download:
            directive = ("Задайте новую Метку или выберите из существующих")

            file_types = thumbnailModel.getNoFilesJobCodeNeeded()
            details = file_types.file_types_present_details(title_case=False)
            if sum(file_types.values()) == 1:
                file_details = (
                    (
                        "Метка будет применена к %s, еще не имеющему Метки."
                    )
                    % details
                )
            else:
                file_details = (
                    (
                        "Метка будет применена к %s, еще не имеющим Метки."
                    )
                    % details
                )

            hint = (
                "<b>Hint:</b> To assign Job Codes before the download begins, select "
                "photos or videos and apply a new or existing Job Code to them via the "
                "Job Code panel."
            )
            file_details = f"{file_details}<br><br><i>{hint}</i>"

            title = ("Применить Метку к Загрузке")
        else:
            directive = ("Задайте новую Метку")

            file_types = thumbnailModel.getNoFilesSelected()
            no_selection_made = sum(file_types.values()) == 0
            if no_selection_made:
                file_details = (
                    "<i>"
                    + (
                        "<b>Подсказка:</b> Перед вводом нового Кода Задачи выберите фото или видео и "
                        "он применится к ним."
                    )
                    + "</i>"
                )

            else:
                details = file_types.file_types_present_details(title_case=False)
                file_details = (
                    "<i>"
                    + ("Новая Метка будет применена к %s.") % details
                    + "</i>"
                )

            title = ("Новая Метка")

        instructionLabel = QLabel(f"<b>{directive}</b><br><br>{file_details}<br>")
        instructionLabel.setWordWrap(True)

        self.jobCodeComboBox = QComboBox()
        self.jobCodeComboBox.addItems(job_codes)
        self.jobCodeComboBox.setEditable(True)

        exp = "[^/\\0]+" if not self.prefs.strip_characters else '[^\\:\\*\\?"<>|\\0/]+'

        self.jobCodeExp = QRegularExpression()
        self.jobCodeExp.setPattern(exp)
        self.jobCodeValidator = QRegularExpressionValidator(
            self.jobCodeExp, self.jobCodeComboBox
        )
        self.jobCodeComboBox.setValidator(self.jobCodeValidator)

        if not on_download:
            self.jobCodeComboBox.clearEditText()

        if self.prefs.job_code_sort_key == 0:
            if self.prefs.job_code_sort_order == 0:
                self.jobCodeComboBox.setInsertPolicy(QComboBox.InsertAtTop)
            else:
                self.jobCodeComboBox.setInsertPolicy(QComboBox.InsertAtBottom)
        else:
            self.jobCodeComboBox.setInsertPolicy(QComboBox.InsertAlphabetically)

        icon = QIcon(data_file_path("smartvision-guard.svg")).pixmap(
            standardIconSize()
        )
        iconLabel = QLabel()
        iconLabel.setPixmap(icon)
        iconLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
        iconLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)

        jobCodeLabel = QLabel(("Метка:"))
        jobCodeLabel.setBuddy(self.jobCodeComboBox)

        if on_download or not no_selection_made:
            self.rememberCheckBox = QCheckBox(("Запомнить этот Код Задачи"))
            self.rememberCheckBox.setChecked(parent.prefs.remember_job_code)

        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        translateDialogBoxButtons(buttonBox)

        grid = QGridLayout()
        grid.addWidget(iconLabel, 0, 0, 4, 1)
        grid.addWidget(instructionLabel, 0, 1, 1, 2)
        grid.addWidget(jobCodeLabel, 1, 1)
        grid.addWidget(self.jobCodeComboBox, 1, 2)

        if hasattr(self, "rememberCheckBox"):
            grid.addWidget(self.rememberCheckBox, 2, 1, 1, 2)
            grid.addWidget(buttonBox, 3, 0, 1, 3)
        else:
            grid.addWidget(buttonBox, 2, 0, 1, 3)

        grid.setColumnStretch(2, 1)
        self.setLayout(grid)
        self.setWindowTitle(title)

        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)

    @pyqtSlot()
    def accept(self) -> None:
        self.job_code = self.jobCodeComboBox.currentText()
        if hasattr(self, "rememberCheckBox"):
            self.remember = self.rememberCheckBox.isChecked()
            self.sdrApp.prefs.remember_job_code = self.remember
        else:
            self.remember = True
        super().accept()


class JobCodeOptionsWidget(FlexiFrame):
    """
    Display and allow editing of Job Codes.
    """

    def __init__(self, prefs: Preferences, sdrApp, parent) -> None:
        super().__init__(parent=parent)

        self.sdrApp = sdrApp
        self.prefs = prefs

        self.setBackgroundRole(QPalette.Base)
        self.setAutoFillBackground(True)

        self.file_selected = False
        self.prompting_for_job_code = False

        jobCodeLayout = QGridLayout()
        layout = self.layout()
        layout.addLayout(jobCodeLayout)
        self.setLayout(layout)

        self.messageWidget = MessageWidget(
            (
                (
                    "Выберите фото и видео, к которым можно применить новую или существующую "
                    "Метку."
                ),
                (
                    "Новая Метка будет применена ко всем выбранным фото и/или видео."
                ),
                (
                    "Нажмите кнопку Применить, чтобы применить текущую Метку ко всем выбранным "
                    "фото и/или видео. Также вы можете просто дважды кликнуть по Метке."
                ),
                (
                    "Удаление Метки удалит ее только из списка сохраненных Меток, но не из фото "
                    "или видео, к которым она могла быть применена."
                ),
                (
                    "Если вы хотите использовать Метки, настройте переименование файлов или "
                    "создание имен подпапок с их использованием."
                ),
            )
        )

        self.setDefaultMessage()

        self.sortCombo = ChevronCombo(in_panel=True)
        self.sortCombo.addItem(("Последние"), JobCodeSort.last_used)
        self.sortCombo.addItem(("Метка"), JobCodeSort.code)
        if self._sort_index_valid(self.prefs.job_code_sort_key):
            self.sortCombo.setCurrentIndex(self.prefs.job_code_sort_key)
        self.sortCombo.currentIndexChanged.connect(self.sortComboChanged)
        self.sortLabel = self.sortCombo.makeLabel(("Сортировка Меток:"))

        self.sortOrder = ChevronCombo(in_panel=True)
        self.sortOrder.addItem(("По возрастанию"), Qt.AscendingOrder)
        self.sortOrder.addItem(("По убыванию"), Qt.DescendingOrder)
        if self._sort_index_valid(self.prefs.job_code_sort_order):
            self.sortOrder.setCurrentIndex(self.prefs.job_code_sort_order)
        self.sortOrder.currentIndexChanged.connect(self.sortOrderChanged)

        font: QFont = self.font()
        font.setPointSize(font.pointSize() - 2)
        for widget in (self.sortLabel, self.sortCombo, self.sortOrder):
            widget.setFont(font)

        self.newButton = MessageButton(("Новая..."))
        self.newButton.isActive.connect(self.newButtonActive)
        self.newButton.isInactive.connect(self.setDefaultMessage)
        self.newButton.clicked.connect(self.newButtonClicked)
        self.applyButton = MessageButton(("Применить"))
        self.applyButton.isActive.connect(self.applyButtonActive)
        self.applyButton.isInactive.connect(self.setDefaultMessage)
        self.applyButton.clicked.connect(self.applyButtonClicked)
        self.removeButton = MessageButton(("Удалить"))
        self.removeButton.isActive.connect(self.removeButtonActive)
        self.removeButton.isInactive.connect(self.setDefaultMessage)
        self.removeButton.clicked.connect(self.removeButtonClicked)
        self.removeAllButton = MessageButton(("Удалить все"))
        self.removeAllButton.isActive.connect(self.removeButtonActive)
        self.removeAllButton.isInactive.connect(self.setDefaultMessage)
        self.removeAllButton.clicked.connect(self.removeAllButtonClicked)

        self.jobCodesWidget = QNarrowListWidget()
        self.jobCodesWidget.currentRowChanged.connect(self.rowChanged)
        self.jobCodesWidget.itemDoubleClicked.connect(self.rowDoubleClicked)
        self.jobCodesWidget.setSelectionMode(QAbstractItemView.SingleSelection)
        self.jobCodesWidget.setSizePolicy(
            QSizePolicy.MinimumExpanding, QSizePolicy.MinimumExpanding
        )

        if self.prefs.list_not_empty("job_codes"):
            self._insertJobCodes(job_code=self.prefs.job_codes[0], clear=False)

        sortLayout = QHBoxLayout()
        sortLayout.addWidget(self.sortLabel)
        sortLayout.addWidget(self.sortCombo)
        sortLayout.addWidget(self.sortOrder)
        sortLayout.addStretch()

        jobCodeLayout.addWidget(self.jobCodesWidget, 0, 0, 1, 2)
        jobCodeLayout.addLayout(sortLayout, 1, 0, 1, 2)
        jobCodeLayout.addWidget(self.messageWidget, 2, 0, 1, 2)
        jobCodeLayout.addWidget(self.newButton, 3, 0, 1, 1)
        jobCodeLayout.addWidget(self.applyButton, 3, 1, 1, 1)
        jobCodeLayout.addWidget(self.removeButton, 4, 0, 1, 1)
        jobCodeLayout.addWidget(self.removeAllButton, 4, 1, 1, 1)

        self.setSizePolicy(QSizePolicy.MinimumExpanding, QSizePolicy.Expanding)

        self.setWidgetStates()

    def _sort_index_valid(self, index: int) -> bool:
        return index in (0, 1)

    def _jobCodes(self) -> list[str]:
        """
        :return: list of job codes sorted according to user-specified
         criteria
        """
        reverse = self.sortOrder.currentIndex() == 1
        if self.sortCombo.currentIndex() == 1:
            return sorted(self.prefs.job_codes, key=str.lower, reverse=reverse)
        if reverse:
            return list(reversed(self.prefs.job_codes))
        return self.prefs.job_codes

    def _insertJobCodes(self, job_code: str = None, clear=True) -> None:
        """
        Insert job codes into list widget according to the sort order
        specified by the user.

        If no job codes exist, does nothing.

        Alternative to clearing the widget and using python to sort the
        list of job codes would be to implement __lt__ in QListWidgetItem,
        and turn on QListWidget sorting. The code as implemented strikes
        me as simpler.

        :param job_code: job_code to set current row to. If not specified,
         the current row is used.
        :param clear:
        :return:
        """
        if not self.prefs.list_not_empty("job_codes"):
            return

        if job_code is None:
            row = self.jobCodesWidget.currentRow()
            if row >= 0:
                job_code = self.jobCodesWidget.item(row).text()

        if clear:
            self.jobCodesWidget.clear()

        logging.debug(
            "Inserting %s job codes into job code widget", len(self.prefs.job_codes)
        )
        job_codes = self._jobCodes()
        self.jobCodesWidget.insertItems(0, job_codes)

        if job_code is not None:
            self.jobCodesWidget.setCurrentRow(job_codes.index(job_code))
        else:
            self.jobCodesWidget.setCurrentRow(0)

    @pyqtSlot(int)
    def sortComboChanged(self, index: int) -> None:
        if index >= 0:
            self._insertJobCodes()
            self.prefs.job_code_sort_key = index

    @pyqtSlot(int)
    def sortOrderChanged(self, index: int) -> None:
        if index >= 0:
            self._insertJobCodes()
            self.prefs.job_code_sort_order = index

    @pyqtSlot()
    def newButtonActive(self) -> None:
        if self.prefs.any_pref_uses_job_code():
            if self.file_selected:
                self.messageWidget.setCurrentIndex(2)
            else:
                self.messageWidget.setCurrentIndex(1)

    @pyqtSlot()
    def applyButtonActive(self) -> None:
        if self.prefs.any_pref_uses_job_code():
            if self.file_selected:
                self.messageWidget.setCurrentIndex(3)
            else:
                self.messageWidget.setCurrentIndex(1)

    @pyqtSlot()
    def removeButtonActive(self) -> None:
        if self.prefs.any_pref_uses_job_code():
            self.messageWidget.setCurrentIndex(4)

    @pyqtSlot()
    def setDefaultMessage(self) -> None:
        if self.prefs.any_pref_uses_job_code():
            if not self.file_selected:
                self.messageWidget.setCurrentIndex(1)
            else:
                self.messageWidget.setCurrentIndex(0)
        else:
            self.messageWidget.setCurrentIndex(5)

    @pyqtSlot(int)
    def rowChanged(self, row: int) -> None:
        self.setWidgetStates()

    @pyqtSlot(QListWidgetItem)
    def rowDoubleClicked(self, item: QListWidgetItem) -> None:
        if self.file_selected:
            assert self.applyButton.isEnabled()
            self.applyButtonClicked()

    @pyqtSlot()
    def setWidgetStates(self) -> None:
        """
        Set buttons enable or disable depending on selections, and updates
        the message widget contents.
        """

        job_code_selected = self.jobCodesWidget.currentRow() >= 0
        self.file_selected = self.sdrApp.anyFilesSelected()

        self.newButton.setEnabled(True)
        self.applyButton.setEnabled(job_code_selected and self.file_selected)
        self.removeButton.setEnabled(job_code_selected)
        self.removeAllButton.setEnabled(self.prefs.list_not_empty("job_codes"))
        self.setDefaultMessage()

    @pyqtSlot()
    def applyButtonClicked(self) -> None:
        row = self.jobCodesWidget.currentRow()
        if row < 0:
            logging.error(
                "Did not expect Apply Job Code button to be enabled when no Job Code "
                "is selected."
            )
            return

        try:
            job_code = self.jobCodesWidget.item(row).text()
        except Exception:
            logging.exception(
                "Job Code did not exist when obtaining its value from the list widget"
            )
            return

        self.sdrApp.applyJobCode(job_code=job_code)

        try:
            self.prefs.del_list_value(key="job_codes", value=job_code)
        except KeyError:
            logging.exception(
                "Attempted to delete non existent value %s from Job Codes while in "
                "process of moving it to the front of the list",
                job_code,
            )
        self.prefs.add_list_value(key="job_codes", value=job_code)

        if self.sortCombo.currentIndex() != 1:
            self._insertJobCodes(job_code=job_code)

    @pyqtSlot()
    def removeButtonClicked(self) -> None:
        row = self.jobCodesWidget.currentRow()
        item: QListWidgetItem = self.jobCodesWidget.takeItem(row)
        try:
            self.prefs.del_list_value(key="job_codes", value=item.text())
        except KeyError:
            logging.exception(
                "Attempted to delete non existent value %s from Job Codes", item.text()
            )

    @pyqtSlot()
    def removeAllButtonClicked(self) -> None:
        message = ("Вы действительно хотите удалить все Метки?")
        msgBox = standardMessageBox(
            parent=self,
            title=("Удалить все Метки"),
            message=message,
            rich_text=False,
            standardButtons=QMessageBox.Yes | QMessageBox.No,
        )
        if msgBox.exec() == QMessageBox.Yes:
            # Must clear the job codes before adjusting the qlistwidget,
            # or else the Remove All button will not be disabled.
            self.prefs.job_codes = [""]
            self.jobCodesWidget.clear()

    @pyqtSlot()
    def newButtonClicked(self) -> None:
        self.getJobCode(on_download=False)

    def getJobCode(self, on_download: bool) -> bool:
        if not self.prompting_for_job_code:
            logging.debug("Prompting for job code")
            self.prompting_for_job_code = True
            dialog = JobCodeDialog(
                self.sdrApp, on_download=on_download, job_codes=self._jobCodes()
            )
            if dialog.exec():
                self.prompting_for_job_code = False
                logging.debug("Job code entered / selected")
                job_code = dialog.job_code
                if job_code:
                    if dialog.remember:
                        # If the job code is already in the
                        # preference list, delete it
                        job_codes = self.sdrApp.prefs.job_codes.copy()
                        while job_code in job_codes:
                            job_codes.remove(job_code)
                        # Add the just chosen / entered Job Code to the front
                        self.sdrApp.prefs.job_codes = [job_code] + job_codes
                        self._insertJobCodes(job_code=job_code)
                    if not on_download:
                        self.sdrApp.applyJobCode(job_code=job_code)
                    else:
                        self.sdrApp.thumbnailModel.assignJobCodesToMarkedFilesWithNoJobCode(
                            job_code=job_code
                        )
                    return True
            else:
                self.prompting_for_job_code = False
                logging.debug("No job code entered or selected")
        else:
            logging.debug("Not prompting for job code, because already doing so")
        return False


class JobCodePanel(ScrollAreaNoFrame):
    """
    JobCode preferences widget
    """

    def __init__(self, parent) -> None:
        super().__init__(name="jobCodePanel", parent=parent)
        assert parent is not None
        self.sdrApp = parent
        self.prefs = self.sdrApp.prefs
        self.setObjectName("jobCodePanelScrollArea")

        self.jobCodePanel = QPanelView(
            label=("Метки"),
        )
        self.jobCodePanel.setObjectName("jobCodePanel")
        self.jobCodeOptions = JobCodeOptionsWidget(
            prefs=self.prefs, sdrApp=self.sdrApp, parent=self
        )
        self.jobCodeOptions.setObjectName("jobCodeOptions")

        self.jobCodePanel.addWidget(self.jobCodeOptions)
        self.verticalScrollBarVisible.connect(
            self.jobCodeOptions.containerVerticalScrollBar
        )
        self.horizontalScrollBarVisible.connect(
            self.jobCodeOptions.containerHorizontalScrollBar
        )

        widget = QWidget()
        layout = QVBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        widget.setLayout(layout)
        layout.addWidget(self.jobCodePanel)
        self.setWidget(widget)
        self.setWidgetResizable(True)
        self.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Expanding)

        self.sdrApp.thumbnailView.selectionModel().selectionChanged.connect(
            self.jobCodeOptions.setWidgetStates
        )
        self.sdrApp.thumbnailModel.selectionReset.connect(
            self.jobCodeOptions.setWidgetStates
        )

    def needToPromptForJobCode(self) -> bool:
        return (
            self.prefs.any_pref_uses_job_code()
            and self.sdrApp.thumbnailModel.jobCodeNeeded()
        )

    def getJobCodeBeforeDownload(self) -> bool:
        """
        :return: True if job code was entered and applied
        """
        return self.jobCodeOptions.getJobCode(on_download=True)

    def updateDefaultMessage(self) -> None:
        self.jobCodeOptions.setDefaultMessage()
