#
#

"""
Show 'Did you know?' dialog at start-up
"""

from PyQt5.QtCore import QSettings, QSize, Qt, pyqtSlot
from PyQt5.QtGui import (
    QCloseEvent,
    QFont,
    QFontMetrics,
    QIcon,
    QPixmap,
    QShowEvent,
    QTextCursor,
)
from PyQt5.QtWidgets import (
    QApplication,
    QCheckBox,
    QDialog,
    QDialogButtonBox,
    QHBoxLayout,
    QLabel,
    QPushButton,
    QTextBrowser,
    QVBoxLayout,
)

from svg.internationalisation.install import install_gettext
from svg.prefs.preferences import Preferences
from svg.tools.utilities import data_file_path
from svg.ui.viewutils import translateDialogBoxButtons

install_gettext()

tips = (
    (
        ("Кликните по флажку файла чтобы отметить его для загрузки или снять отметку."),
        "tips/marksingle.png",
    ),
    (
        (
            "Файлы, которые уже были загружены запоминаются. Вы по-прежнему можете снова "
            "отметить уже загруженные файлы для загрузки, но по умолчанию отметки с них "
            "сняты, и их миниатюры слегка затуманены чтобы вы могли их отличить от "
            "файлов, которые еще будут загружаться."
        ),
        "tips/previouslydownloaded.png",
    ),
    (
        (
            "Если выбрано более одного файла, независимо от того были они отмечены или "
            "нет они все примут отметку файла, по флажку которого вы кликните."
        ),
        "tips/markmany.png",
    ),
    (
        (
            "Кликните по флажку устройства, чтобы быстро отметить все его файлы для "
            "загрузки или снять с них отметку."
        ),
        "tips/markall.png",
    ),
    (
        (
            "После завершения загрузки флажок будет заменен значком. Цвет значка "
            "показывает была ли загрузка успешной (зеленый), были ли проблемы с "
            "переименованием (желтый) или загрузка не удалась (красный)."
        ),
        "tips/downloaded.png",
    ),
    (
        (
            "При наличии каких-либо проблем в нижней части окна появится красный значок, "
            "отображающий количество возникших ошибок. Нажатие на него откроет окно "
            "Отчета об ошибках."
        ),
        "tips/errorreporticon.png",
        (
            "Окно Отчета об ошибках отображает список ошибок, возникших до, во время ил "
            "после загрузки. Оранжевые треугольники отмечают предупреждения, красные "
            "кружки - сбои, а черные кружки - более серьезные сбои. Вы можете нажать на "
            "ссылку, чтобы открыть ее файл или устройство в файл менеджере. Вы также "
            "можете выполнять поиск в отчете, используя поле поиска в нижней левой части "
            "окна Отчета об ошибках."
        ),
        "tips/errorreport.png",
    ),
    (
        (
            "Вы можете одновременно выполнять загрузку с нескольких камер, смартфонов, "
            "карт памяти и жестких дисков&mdash;с любого количества устройств, которое "
            "ваша система сможет обслуживать одновременно."
        ),
        "tips/multipledevices.png",
    ),
    (
        (
            "<b>Шкала времени</b> группирует фото и видео на основании того, сколько "
            "времени прошло между снимками. Вы можете использовать ее для определения "
            "фото и видео, снятых в разные периоды одного дня или за несколько дней "
            "подряд."
        ),
        "tips/timeline.png",
        (
            "<p>На иллюстрации выше первая строка шкалы времени окрашена черным, потому "
            "что все файлы за эту дату уже были загружены.</p><p>Ползунок Шкалы времени "
            "определяет прошедшее между снимками время, которое используется для "
            "построения Шкалы времени:</p>"
        ),
        "tips/timelineadjust.png",
    ),
    (
        (
            "Для просмотра фото и видео за определенный период времени используйте мышь "
            "(при необходимости в сочетании с клавишами <tt>Shift</tt> и <tt>Ctrl</tt>) "
            "для выбора временных интервалов. Когда период времени выбран, кнопка Шкалы "
            "времени с левой стороны главного окна будет подсвечена."
        ),
        "tips/timelineselect.png",
        (
            "В загрузку всегда включаются все файлы, отмеченные для загрузки, включая и "
            "те, которые в данный момент не отображаются по причине использования Шкалы "
            "времени."
        ),
    ),
    (
        (
            "Вы можете скрывать или отображать источники загрузки кликая по имени "
            "устройства с которого вы загружаете в окне программы вверху слева."
        ),
        "tips/deviceshidden.png",
    ),
    (
        (
            "Миниатюры могут быть отсортированы с использованием:<ol><li><b>Времени "
            "изменения:</b> когда файл был изменен последний раз, согласно его метаданным "
            "(где возможно) или согласно данным файловой системы.</li><li><b>Состояния "
            "отметки:</b> отмечен ли файл для загрузки.</li><li><b>Имени файла:</b> "
            "полное имя файла, включая расширение.</li><li><b>Расширения:</b> расширение "
            "файла. Вы можете использовать это для группировки jpeg и raw файлов, "
            "например.</li><li><b>Типа файла:</b> фото или видео.</li><li><b>Устройства:</"
            "b> имя устройства, с которого будут загружаться фото и видео.</li></ol>"
        ),
        "tips/thumbnailsort.png",
    ),
    (
        (
            "Одна из наиболее полезных возможностей SmartVision Guard - это способность "
            "автоматически генерировать подпапки и переименовывать файлы во время "
            "загрузки, используя схему по вашему выбору."
        ),
        "tips/downloadwhereandrename.png",
        (
            "Чтобы указать, куда вы хотите загружать файлы и как вы хотите, чтобы они "
            "назывались, откройте соответствующую панель с правой стороны окна "
            "приложения: <b>Место хранения</b>, <b>Переименование</b>, или <b>Метка</b>."
        ),
    ),
    (
        (
            "Когда вы думаете о структуре папок загрузки, примите во внимание два разных "
            "типа папок:<ol><li>The <b>место хранения</b>, т.е. &quot;Изображения&quot;, "
            "&quot;Фото&quot;, или &quot;Видео&quot;. Эта папка уже должна существовать в "
            "вашем компьютере. На иллюстрации ниже, места хранения &quot;Pictures&quot; и "
            "&quot;Videos&quot;. Имя места хранения отображается на серой панели над "
            "деревом папок, со значком папки слева от него и значком шестеренки с правого "
            "края.</li><li>The <b>подпапки загрузки</b>, которые являются папками, "
            "автоматически созданными SmartVision Guard. Они не должны обязательно уже "
            "существовать на вашем компьютере, но если они уже есть, то в этом нет ничего "
            "страшного. Они будут созданы внутри места хранения.</li></ol>"
        ),
        "tips/defaultdownloaddirectory.png",
        (
            "Вы можете загружать фото и видео в одно место хранения, или указать разные "
            "места хранения. То же самое касается и подпапок для загрузки фото и "
            "видео&mdash; загружать фото и видео в одни и те же подпапки, или же "
            "использовать разные схемы для каждого типа."
        ),
    ),
    (
        (
            "Автоматически созданные подпапки загрузки могут содержать вложенные "
            "автоматически созданные подпапки, если необходимо. Обычная схема - создание "
            "папки года и внутри нее серии подпапок год-месяц-день."
        ),
        "tips/downloadsubfolders.png",
    ),
    (
        (
            "Где возможно, программа отображает подпапки загрузки для загружаемых фото и "
            "видео:<ol><li>Дерево папок места хранения отображает уже существующие на "
            "вашем компьютере подпапки (имена обычным шрифтом, не курсивом), и подпапки, "
            "которые будут созданы при загрузке (имена курсивом).</li><li>Дерево папок "
            "также показывает, в какую подпапку будут загружены файлы (окрашены в черный "
            "цвет).</li></ol>"
        ),
        "tips/downloadsubfolders.png",
    ),
    (
        (
            "Имена подпапок загрузки обычно создаются с использованием следующих "
            "элементов:<ol><li><b>Метаданных файла</b>, очень часто содержат дату "
            "создания фото или видео, но могут также содержать название камеры, ее "
            "серийный номер, или расширение файла т.е. JPG или CR2.</li><li>A <b>Метки</"
            "b>, любого текста, который вы задаете во время загрузки, такого как название "
            "события или места.</li><li><b>Текста</b> который вы хотите, чтобы появлялся "
            "всегда, как тире или пробел.</li></ol> Создание имен подпапок из года, "
            "следующего за ним месяца, и, напоследок, дня в цифровом виде упрощает их "
            "сортировку в файл менеджере, и поэтому выбрано по умолчанию:"
        ),
        "tips/downloadsubfolders.png",
    ),
    (
        (
            "Для автоматического создания подпапок при загрузке, вы можете использовать "
            "одну из встроенных предустановок SmartVision Guard, или создать свою. "
            "Кликните по значку с шестеренкой для открытия выпадающего меню:"
        ),
        "tips/subfoldermenu.png",
        (
            "Используя выпадающее меню, выберите встроенную предустановку или кликните на "
            "<b>Своя</b> для задания своей схемы. Вы создадите свою схему, используя "
            "Редактор создания подпапок для фото и видео:"
        ),
        "tips/subfoldergeneration.png",
    ),
    (
        (
            "Очень легко загружать raw файлы в одну папку, а jpeg файлы - в другую. "
            "Просто используйте <b>Расширение файла</b> как часть схемы создания подпапок "
            "загрузки:"
        ),
        "tips/subfoldergenerationext.png",
        (
            "На этой иллюстрации показана сохраненная предустановка с именем &quot;My "
            "custom preset&quot;."
        ),
    ),
    (
        (
            "Вы не обязаны создавать вложенные подпапки загрузки. Эта иллюстрация "
            "показывает создание подпапок загрузки, содержащих только дату, когда сделан "
            "снимок и метку:"
        ),
        "tips/subfoldergeneration.png",
    ),
    (
        (
            "Несмотря на то, что есть много встроеных схем переименования с "
            "использованием даты/времени, вы можете решить, что вам надо что-то другое. "
            "Вы можете без проблем создать свою схему. Вы можете комбинировать варианты "
            "даты/времени, чтобы создать новые комбинации. Предположим вам нужен формат "
            "даты, содержащий год (ГГГГ), дефис и месяц (ММ) для получения ГГГГ-ММ. Вы "
            "можете создать его как показано здесь (обратите внимание на красный кружок "
            "вокруг дефиса:"
        ),
        "tips/customdate.png",
        (
            "Прочесть больше о всех способах создания имен файлов и подпапок загрузки вы "
            'можете в <a href=\"http://sdr-technology.ru">документации в '
            "сети</a>."
        ),
    ),
    (
        (
            "<b>Метки</b> позволяют вам легко задать текст, описывающий набор фото и "
            "видео. Вы можете использовать их в именах файлов и подпапок. На этой "
            "иллюстрации к некоторым файлам применена метка &quot;Street&quot;, а "
            "выбранные файлы должны получить метку &quot;Green Bazaar&quot;:"
        ),
        "tips/jobcodes.png",
        (
            "Вы можете применять новые или существующие Метки перед началом загрузки. "
            "Если среди загружаемых файлов есть такие, к которым еще не была применена "
            "Метка, вам будет предложено задать Метку для этих файлов перед тем как "
            "начнется загрузка."
        ),
    ),
    (
        ("Ищите подсказки, которые направят вас при работе с Метками:"),
        "tips/jobcodehint.png",
        (
            "Подсказки будут зависеть от контекста, например когда наводите мышь на "
            "кнопку."
        ),
    ),
    (
        (
            "Если вы дадите вашим файлам фото и видео уникальные имена, вы никогда не "
            "будете в них путиться. Настоятельно рекомендуется для задания уникальных "
            "имен файлов использовать <b>Нумерацию</b>!"
        ),
        "tips/photoeditordefault.png",
        (
            "<p>Для того, чтобы помочь вам в назначении уникальных имен вашим фото и "
            "видео есть четыре вида нумерации:<ol><li><b>Загружено сегодня</b>: "
            "подсчитывает завершенные загрузки в течение дня.</li><li><b>Сквозная "
            "нумерация</b>: похожа на Загружено сегодня, но ее значение сохраняется с "
            "прошлого запуска программы.</li><li><b>Сеансовая нумерация</b>: сбрасывается "
            "при каждом запуске программы.</li><li><b>Символьная нумерация</b>: как "
            "сеансовая нумерация, но использует буквы.</li></ol></p>"
        ),
    ),
    (
        (
            "Панель <b>Переименование</b> позволяет настроить переименование файлов. Для "
            "переименования файлов вы можете выбрать одну из существующих предустановок "
            "переименования или задать свои правила."
        ),
        "tips/renameoptions.png",
        (
            "<p>Параметр <b>Синхронизировать RAW + JPEG</b> если вы используете функцию "
            "RAW + JPEG в вашей камере и вы используете нумерацию при переименовании "
            "ваших фото. Включение этого параметра приводит к тому, что программа "
            "определяет пары RAW и JPEG файлов, и, после их определния, к их именам "
            "применяется одинаковая нумерация. Более того, нумерация изменяется, как если "
            "бы фото было одно.</p>"
        ),
    ),
    (
        (
            "Во время загрузки вы можете создавать резервные копии ваших фото и видео в "
            "нескольких местах, таких как внешние жесткие диски или сетевые хранилища. "
            "Устройства для хранения резервных копий могут быть определены автоматически "
            "или вы можете точно задать место хранения резервных копий."
        ),
        "tips/backup.png",
        (
            "В этом примере накопитель <b>photobackup</b> не содержит папки с именем "
            "<tt>Videos</tt>, поэтому резервное копирование видео на него выполнено не "
            "будет."
        ),
    ),
    (
        (
            "Некоторые параметры программы могут быть заданы из командной строки, включая "
            "источники загрузки, места хранения и места хранения резервных копий. "
            "Дополнительно, настройки могут быть сброшены на значения по умолчанию, и "
            "может быть очищен кэш и запомненные файлы."
        )
        + ("Также вы можете импортировать настройки из старой версии программы."),
        "tips/commandline.png",
    ),
    (
        (
            "SmartVision Guard работает с тремя видами кэша:<ol><li>A <b>кэшем миниатюр</"
            "b>, назначение которого - хранить миниатюры файлов с фаших камер, карт "
            "памяти и других устройств.</li><li>A <b>временным кэшем</b> файлов, "
            "загруженных с камеры, одним для фото и другим для видео. Они находятся во "
            "временных подпапках в месте хранения.</li><li><b>кэшем миниатюр рабочего "
            "стола</b>, в который SmartVision Guard сохраняет миниатюры загруженных RAW и "
            "TIFF файлов. Файл менеджеры используют этот кэш, что значит, что они будут "
            "отображать миниатюры для этих файлов. </li></ol>"
        ),
    ),
)


# To add, possibly:
# Ignoring Devices
# Don't access camera from another program
# Device Scanning prefs
# Ignored Paths
# Automation
# Error Handling Preferences
# Miscellaneous Preferences


class Tips:
    def __getitem__(self, item) -> str:
        if 0 > item >= len(tips):
            item = 0
        tip = tips[item]
        text = ""
        for idx, value in enumerate(tip):
            if idx % 2 == 0:
                if not value.startswith("<p>"):
                    text = f"{text}<p>{value}</p><p></p>"
                else:
                    text = f"{text}{value}<p></p>"
            else:
                text = f'{text}<img src="{data_file_path(value)}">'
        return text

    def __len__(self):
        return len(tips)


class DidYouKnowDialog(QDialog):
    def __init__(self, prefs: Preferences, parent=None) -> None:
        super().__init__(parent)
        self.sdrApp = parent
        self.prefs = prefs

        self.setWindowTitle(("Совет дня"))

        self.setSizeGripEnabled(True)

        titleFont = QFont()
        titleFont.setPointSize(titleFont.pointSize() + 3)
        pixsize = int(QFontMetrics(QFont()).height() * 1.75)

        title = QLabel(("Знаете ли вы...?"))
        title.setFont(titleFont)
        pixmap: QPixmap = QIcon(data_file_path("tips/did-you-know.svg")).pixmap(
            QSize(pixsize, pixsize)
        )

        icon = QLabel()
        icon.setPixmap(pixmap)
        titleLayout = QHBoxLayout()
        titleLayout.addWidget(icon)
        titleLayout.addWidget(title)
        titleLayout.addStretch()

        self.text = QTextBrowser()
        self.text.setOpenExternalLinks(True)
        self.text.setViewportMargins(10, 10, 10, 10)
        self.text.setStyleSheet(
            """
        QTextEdit { background: palette(base); }
        """
        )

        self.text.document().setDefaultStyleSheet(
            """
            b {color: grey;}
            tt {color: darkRed; font-weight: bold;}
            """
        )

        self.tips = Tips()

        self.showTips = QCheckBox(("Показывать советы при запуске"))
        self.showTips.setChecked(self.prefs.did_you_know_on_startup)
        self.showTips.stateChanged.connect(self.showTipsChanged)

        self.nextButton = QPushButton(("Далее"))
        self.previousButton = QPushButton(("Назад"))

        buttons = QDialogButtonBox(QDialogButtonBox.Close)
        translateDialogBoxButtons(buttons)
        buttons.addButton(self.previousButton, QDialogButtonBox.ActionRole)
        buttons.addButton(self.nextButton, QDialogButtonBox.ActionRole)
        self.previousButton.clicked.connect(self.previousButtonClicked)
        self.nextButton.clicked.connect(self.nextButtonClicked)
        buttons.rejected.connect(self.close)

        layout = QVBoxLayout()
        self.setLayout(layout)

        layout.addLayout(titleLayout)
        layout.addWidget(self.text)
        layout.addWidget(self.showTips)
        layout.addWidget(buttons)

        settings = QSettings()
        settings.beginGroup("DidYouKnowWindow")

        default_width = 570
        default_height = 350

        size = settings.value("windowSize", QSize(default_width, default_height))

        settings.endGroup()
        self.resize(size)

        self.showTip()

    def incrementTip(self) -> None:
        if self.prefs.did_you_know_index + 1 == len(self.tips):
            self.prefs.did_you_know_index = 0
        else:
            self.prefs.did_you_know_index = self.prefs.did_you_know_index + 1

    def decrementTip(self) -> None:
        if self.prefs.did_you_know_index == 0:
            self.prefs.did_you_know_index = len(self.tips) - 1
        else:
            self.prefs.did_you_know_index = self.prefs.did_you_know_index - 1

    def showTip(self) -> None:
        self.text.clear()
        self.text.append(self.tips[self.prefs.did_you_know_index])
        self.text.moveCursor(QTextCursor.Start)

    def showEvent(self, event: QShowEvent) -> None:
        self.nextButton.setDefault(True)
        self.nextButton.setFocus(Qt.OtherFocusReason)
        event.accept()

    @pyqtSlot(int)
    def showTipsChanged(self, state: int) -> None:
        self.prefs.did_you_know_on_startup = state == Qt.Checked

    @pyqtSlot()
    def nextButtonClicked(self) -> None:
        self.incrementTip()
        self.showTip()

    @pyqtSlot()
    def previousButtonClicked(self) -> None:
        self.decrementTip()
        self.showTip()

    @pyqtSlot()
    def activate(self) -> None:
        self.showTip()
        self.setVisible(True)
        self.activateWindow()
        self.raise_()

    def reject(self) -> None:
        """
        Called when user hits escape key
        """

        self.saveSettings()
        if self.sdrApp is None:
            super().reject()

    def closeEvent(self, event: QCloseEvent) -> None:
        self.saveSettings()
        if self.sdrApp is None:
            event.accept()
        else:
            event.ignore()
            self.hide()

    def saveSettings(self) -> None:
        self.incrementTip()
        settings = QSettings()
        settings.beginGroup("DidYouKnowWindow")
        settings.setValue("windowSize", self.size())
        settings.endGroup()


if __name__ == "__main__":
    # Application development test code:

    app = QApplication([])

    app.setOrganizationName("SmartVision Guard")
    app.setOrganizationDomain("sdr-technology.ru")
    app.setApplicationName("SmartVision Guard")

    prefs = Preferences()

    dialog = DidYouKnowDialog(prefs=prefs)
    dialog.show()
    app.exec_()
