#
#

"""
Notify user of problems when downloading: problems with subfolder and filename
generation, download errors, and so forth

Goals
=====

Group problems into tasks:
  1. scanning
  2. copying
  3. renaming (presented to user as finalizing file and download subfolder names)
  4. backing up - per backup device

Present messages in a human-readable manner.
Multiple metadata problems can occur: group them.
Distinguish error severity

"""

import logging
from collections import deque
from collections.abc import Iterator
from html import escape

from svg.camera import gphoto2_named_error
from svg.constants import ErrorType
from svg.internationalisation.utilities import make_internationalized_list


def make_href(name: str, uri: str) -> str:
    """
    Construct a hyperlink.
    """

    # Note: keep consistent with ErrorReport._saveUrls()
    return f'<a href="{uri}">{escape(name)}</a>'


class Problem:
    def __init__(
        self,
        name: str | None = None,
        uri: str | None = None,
        exception: Exception | None = None,
        **attrs,
    ) -> None:
        for attr, value in attrs.items():
            setattr(self, attr, value)
        self.name = name
        self.uri = uri
        self.exception = exception

    @property
    def title(self) -> str:
        logging.critical(
            "title() not implemented in subclass %s", self.__class__.__name__
        )
        return "undefined"

    @property
    def body(self) -> str:
        logging.critical(
            "body() not implemented in subclass %s", self.__class__.__name__
        )
        return "undefined"

    @property
    def details(self) -> list[str]:
        if self.exception is not None:
            try:
                return [
                    escape(("Ошибка: %(errno)s %(strerror)s"))
                    % dict(errno=self.exception.errno, strerror=self.exception.strerror)
                ]
            except AttributeError:
                return [escape(("Ошибка: %s")) % self.exception]
        else:
            return []

    @property
    def href(self) -> str:
        if self.name and self.uri:
            return make_href(name=self.name, uri=self.uri)
        else:
            logging.critical(
                "href() is missing name or uri in subclass %s", self.__class__.__name__
            )

    @property
    def severity(self) -> ErrorType:
        return ErrorType.warning


class SeriousProblem(Problem):
    @property
    def severity(self) -> ErrorType:
        return ErrorType.serious_error


class CameraGpProblem(SeriousProblem):
    @property
    def details(self) -> list[str]:
        try:
            return [
                escape(("Ошибка GPhoto2: %ss"))
                % escape(gphoto2_named_error(self.gp_code))
            ]
        except AttributeError:
            return []


class CameraInitializationProblem(CameraGpProblem):
    @property
    def body(self) -> str:
        return escape(
            (
                "Невозможно инициализировать камеру, возможно из-за ее использования другой "
                "программой. Файлы с нее скопированы не будут."
            )
        )

    @property
    def severity(self) -> ErrorType:
        return ErrorType.critical_error


class CameraDirectoryReadProblem(CameraGpProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно прочесть каталог %s")) % self.href


class CameraFileInfoProblem(CameraGpProblem):
    @property
    def body(self) -> str:
        return (
            escape(("Невозможно получить доступ к времени изменения или размеру для %s")) % self.href
        )


class CameraFileReadProblem(CameraGpProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно прочесть файл %s")) % self.href


class FileWriteProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно записать файл %s")) % self.href


class FileMoveProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно переместить файл %s")) % self.href


class FileDeleteProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно удалить файл %s")) % self.href


class FileCopyProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно скопировать файл %s")) % self.href


class FileZeroLengthProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(("Файл %s с нулевой длиной не будет загружен")) % self.href


class FsMetadataReadProblem(Problem):
    @property
    def body(self) -> str:
        return (
            escape(("Нельзя определить время изменения из файловой системы для %s"))
            % self.href
        )


class FileMetadataLoadProblem(Problem):
    @property
    def body(self) -> str:
        return escape(("Невозможно прочесть метаданные из %s")) % self.href


class FileMetadataLoadProblemNoDownload(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(
            (
                "Невозможно загрузить метаданные из %(name)s. Файл %(filetype)s не был "
                "загружен."
            )
        ) % dict(filetype=self.file_type, name=self.href)


class FsMetadataWriteProblem(Problem):
    @property
    def body(self) -> str:
        return (
            escape(
                (
                    "Ошибка установки метаданных файла на файловой системе %s. Если эта ошибка "
                    "повторится на той же файловой системе, повторного отчета не будет."
                )
            )
            % self.href
        )

    @property
    def details(self) -> list[str]:
        return [
            escape(("Ошибка: %(errno)s %(strerror)s"))
            % dict(errno=e.errno, strerror=e.strerror)
            for e in self.mdata_exceptions
        ]


class UnhandledFileProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return (
            escape(("Обнаружен необслуживаемый файл %s. Он не будет загружен."))
            % self.href
        )


class FileAlreadyExistsProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(
            ("%(filetype)s %(destination)s уже существует.")
        ) % dict(filetype=escape(self.file_type_capitalized), destination=self.href)

    @property
    def details(self) -> list[str]:
        d = list()
        d.append(
            escape(
                (
                    "Последнее изменение существующего %(filetype)s %(destination)s было %(date)s "
                    "в %(time)s."
                )
            )
            % dict(
                filetype=escape(self.file_type),
                date=escape(self.date),
                time=escape(self.time),
                destination=self.href,
            )
        )
        d.append(
            escape(
                ("Файл %(filetype)s %(source)s не был загружен с устройства %(device)s.")
            )
            % dict(
                filetype=escape(self.file_type), source=self.source, device=self.device
            )
        )
        return d


class IdentifierAddedProblem(FileAlreadyExistsProblem):
    @property
    def details(self) -> list[str]:
        d = list()
        d.append(
            escape(
                (
                    "Последнее изменение существующего %(filetype)s %(destination)s было %(date)s "
                    "в %(time)s."
                )
            )
            % dict(
                filetype=escape(self.file_type),
                date=escape(self.date),
                time=escape(self.time),
                destination=self.href,
            )
        )
        d.append(
            escape(
                ("Файл %(filetype)s %(source)s был загружен с устройства %(device)s.")
            )
            % dict(
                filetype=escape(self.file_type), source=self.source, device=self.device
            )
        )
        d.append(
            escape(("К имени файла была добавлена уникальная метка '%s'."))
            % self.identifier
        )
        return d

    @property
    def severity(self) -> ErrorType:
        return ErrorType.warning


class BackupAlreadyExistsProblem(FileAlreadyExistsProblem):
    @property
    def details(self) -> list[str]:
        d = list()
        d.append(
            escape(
                (
                    "Последнее изменение существующей резервной копии %(filetype)s "
                    "%(destination)s было %(date)s в %(time)s."
                )
            )
            % dict(
                filetype=escape(self.file_type),
                date=escape(self.date),
                time=escape(self.time),
                destination=self.href,
            )
        )
        d.append(
            escape(
                ("Для файла %(filetype)s %(source)s с устройства %(device)s не была создана резервная копия.")
            )
            % dict(
                filetype=escape(self.file_type), source=self.source, device=self.device
            )
        )
        return d


class BackupOverwrittenProblem(BackupAlreadyExistsProblem):
    @property
    def details(self) -> list[str]:
        d = list()
        d.append(
            escape(
                (
                    "Последнее изменение предыдущей резервной копии %(filetype)s %(destination)s "
                    "было %(date)s в %(time)s."
                )
            )
            % dict(
                filetype=escape(self.file_type),
                date=escape(self.date),
                time=escape(self.time),
                destination=self.name,
            )
        )
        d.append(
            escape(
                (
                    "Для  файла %(filetype)s %(source)s с устройства %(device)s была создана "
                    "резервная копия, перезаписавшая предыдущую резервную копию файла %(filetype)s."
                )
            )
            % dict(
                filetype=escape(self.file_type), source=self.source, device=self.device
            )
        )
        return d

    @property
    def severity(self) -> ErrorType:
        return ErrorType.warning


class DuplicateFileWhenSyncingProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(
            (
                "При синхронизации нумерации RAW + JPEG обнаружен дубликат %(filetype)s "
                "%(file)s, который не был загружен."
            )
        ) % dict(file=self.href, filetype=self.file_type)


class SameNameDifferentExif(Problem):
    @property
    def body(self) -> str:
        return escape(
            (
                "При синхронизации нумерации RAW + JPEG обнаружены фото с теми же именами, но "
                "снятые в другое время:"
            )
        )

    @property
    def details(self) -> list[str]:
        return [
            escape(
                (
                    "%(image1)s было снято %(image1_date)s в %(image1_time)s, и %(image2)s - "
                    "%(image2_date)s в %(image2_time)s."
                )
            )
            % dict(
                image1=self.image1,
                image1_date=self.image1_date,
                image1_time=self.image1_time,
                image2=self.image2,
                image2_date=self.image2_date,
                image2_time=self.image2_time,
            )
        ]


class RenamingAssociateFileProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно завершить имя файла для  %s")) % self.source


class FilenameNotFullyGeneratedProblem(Problem):
    def __init__(
        self,
        name: str | None = None,
        uri: str | None = None,
        exception: Exception | None = None,
        **attrs,
    ) -> None:
        super().__init__(name=name, uri=uri, exception=exception, **attrs)
        self.missing_metadata = []
        self.file_type = ""
        self.destination = ""
        self.source = ""
        self.bad_converstion_date_time = False
        self.bad_conversion_exception: Exception | None = None
        self.invalid_date_time = False
        self.missing_extension = False
        self.missing_image_no = False
        self.component_error = False
        self.component_problem = ""
        self.component_exception = None

    def has_error(self) -> bool:
        """
        :return: True if any of the errors occurred
        """

        return (
            bool(self.missing_metadata)
            or self.invalid_date_time
            or self.bad_converstion_date_time
            or self.missing_extension
            or self.missing_image_no
            or self.component_error
        )

    @property
    def body(self) -> str:
        return escape(
            (
                "Имя файла %(destination)s не было полностью создано для %(filetype)s "
                "%(source)s."
            )
        ) % dict(
            destination=self.destination, filetype=self.file_type, source=self.source
        )

    @property
    def details(self) -> list[str]:
        d = []
        if len(self.missing_metadata) == 1:
            d.append(
                escape(
                    ("Отсутствуют метаданные %(type)s.")
                )
                % dict(type=self.missing_metadata[0])
            )
        elif len(self.missing_metadata) > 1:
            d.append(
                escape(("Отсутствуют следующие метаданные: %s."))
                % make_internationalized_list(self.missing_metadata)
            )

        if self.bad_converstion_date_time:
            d.append(
                escape(("Ошибка преобразования даты/времени: %s."))
                % self.bad_conversion_exception
            )

        if self.invalid_date_time:
            d.append(
                escape(
                    (
                        "Нельзя извлечь корректные метаданные даты/времени или определить время "
                        "изменения файла."
                    )
                )
            )

        if self.missing_extension:
            d.append(escape(("Имя файла не содержит расширения.")))

        if self.missing_image_no:
            d.append(escape(("В имени файла нет порядкового номера.")))

        if self.component_error:
            d.append(
                escape(("Ошибка создания компонента %(component)s. Ошибка: %(error)s"))
                % dict(component=self.component_problem, error=self.component_exception)
            )

        return d


class FolderNotFullyGeneratedProblemProblem(FilenameNotFullyGeneratedProblem):
    @property
    def body(self) -> str:
        return escape(
            (
                "Подпапки загрузки %(folder)s для %(filetype)s %(source)s были созданы только "
                "частично."
            )
        ) % dict(folder=self.destination, filetype=self.file_type, source=self.source)


class NoDataToNameProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(
            (
                "Нет данных для создания %(subfolder_file)s для %(filename)s. Файл "
                "%(filetype)s не загружен."
            )
        ) % dict(
            subfolder_file=self.area,
            filename=self.href,
            filetype=self.file_type,
        )


class RenamingFileProblem(SeriousProblem):
    @property
    def body(self) -> str:
        return escape(
            (
                "Невозможно создать %(filetype)s %(destination)s в %(folder)s. Загружаемый "
                "файл был %(source)s в %(device)s. Он не загружен."
            )
        ) % dict(
            filetype=escape(self.file_type),
            destination=escape(self.destination),
            folder=self.folder,
            source=self.href,
            device=self.device,
        )


class SubfolderCreationProblem(Problem):
    @property
    def body(self) -> str:
        return escape(("Невозможно создать подпапки загрузки %s.")) % self.folder

    @property
    def severity(self) -> ErrorType:
        return ErrorType.critical_error


class BackupSubfolderCreationProblem(SubfolderCreationProblem):
    @property
    def body(self) -> str:
        return escape(("Невозможно создать подпапки для резервных копий %s.")) % self.folder


class Problems:
    def __init__(
        self,
        name: str | None = "",
        uri: str | None = "",
        problem: Problem | None = None,
    ) -> None:
        self.problems = deque()
        self.name = name
        self.uri = uri
        if problem:
            self.append(problem=problem)

    def __len__(self) -> int:
        return len(self.problems)

    def __iter__(self) -> Iterator[Problem]:
        return iter(self.problems)

    def __getitem__(self, index: int) -> Problem:
        return self.problems[index]

    def append(self, problem: Problem) -> None:
        self.problems.append(problem)

    @property
    def title(self) -> str:
        logging.critical(
            "title() not implemented in subclass %s", self.__class__.__name__
        )
        return "undefined"

    @property
    def body(self) -> str:
        return "body"

    @property
    def details(self) -> list[str]:
        return []

    @property
    def href(self) -> str:
        if self.name and self.uri:
            return make_href(name=self.name, uri=self.uri)
        else:
            logging.critical(
                "href() is missing name or uri in %s", self.__class__.__name__
            )


class ScanProblems(Problems):
    @property
    def title(self) -> str:
        return escape(("Проблема сканирования %s")) % self.href


class CopyingProblems(Problems):
    @property
    def title(self) -> str:
        return escape(("Проблема копирования из %s")) % self.href


class RenamingProblems(Problems):
    @property
    def title(self) -> str:
        return escape(
            ("Проблемы с завершением имен файлов и созданием подпапок")
        )


class BackingUpProblems(Problems):
    @property
    def title(self) -> str:
        return escape(("Проблемы создания резервных копий в %s")) % self.href
