#
#

"""
Dialog window to show and manipulate selected user preferences
"""

import logging
import webbrowser

from PyQt5.QtCore import QObject, QSize, Qt, QThread, QTimer, pyqtSignal, pyqtSlot
from PyQt5.QtGui import (
    QCloseEvent,
    QFont,
    QFontMetrics,
    QIcon,
    QMouseEvent,
    QPalette,
    QPixmap,
    QShowEvent,
)
from PyQt5.QtWidgets import (
    QAbstractButton,
    QAbstractItemView,
    QApplication,
    QButtonGroup,
    QCheckBox,
    QComboBox,
    QDialog,
    QDialogButtonBox,
    QFormLayout,
    QGridLayout,
    QGroupBox,
    QHBoxLayout,
    QLabel,
    QLineEdit,
    QListWidgetItem,
    QMessageBox,
    QPushButton,
    QRadioButton,
    QSizePolicy,
    QSpinBox,
    QStackedWidget,
    QStyle,
    QVBoxLayout,
    QWidget,
)

from svg.cache import ThumbnailCacheSql
from svg.constants import (
    CompletedDownloads,
    ConflictResolution,
    KnownDeviceType,
    MarkRawJpeg,
    TreatRawJpeg,
)
from svg.internationalisation.utilities import (
    make_internationalized_list,
    thousands,
)
from svg.metadata.fileextensions import (
    ALL_KNOWN_EXTENSIONS,
    AUDIO_EXTENSIONS,
    PHOTO_EXTENSIONS,
    VIDEO_EXTENSIONS,
    VIDEO_THUMBNAIL_EXTENSIONS,
)
from svg.prefs.preferences import Preferences
from svg.tools.utilities import (
    available_cpu_count,
    available_languages,
    current_version_is_dev_version,
    data_file_path,
    format_size_for_user,
)
from svg.ui.viewutils import (
    QNarrowListWidget,
    StyledLinkLabel,
    darkModePixmap,
    standardMessageBox,
    translateDialogBoxButtons,
)


class ClickableLabel(QLabel):
    clicked = pyqtSignal()

    def mousePressEvent(self, event: QMouseEvent) -> None:
        self.clicked.emit()


CONSOLIDATION_IMPLEMENTED = False
FORCE_EXIFTOOL_VIDEO_IMPLEMENTED = False

system_language = "SYSTEM"


class PreferencesDialog(QDialog):
    """
    Preferences dialog for those preferences that are not adjusted via the main window

    Note:

    When pref value generate_thumbnails is made False, pref values use_thumbnail_cache
    and generate_thumbnails are not changed, even though the preference value shown to
    the user shows False (to indicate that the activity will not occur).
    """

    getCacheSize = pyqtSignal()

    def __init__(self, prefs: Preferences, parent=None) -> None:
        super().__init__(parent=parent)

        self.sdrApp = parent

        self.setWindowTitle(("Настройки"))

        self.prefs = prefs

        self.is_prerelease = current_version_is_dev_version()

        self.panels = QStackedWidget()

        self.chooser = QNarrowListWidget(no_focus_recentangle=True)

        font = QFont()
        fontMetrics = QFontMetrics(font)
        icon_padding = 6
        icon_height = max(fontMetrics.height(), 16)
        icon_width = icon_height + icon_padding
        self.chooser.setIconSize(QSize(icon_width, icon_height))

        palette = QPalette()
        selectedColour = palette.color(palette.HighlightedText)

        if CONSOLIDATION_IMPLEMENTED:
            self.chooser_items = (
                ("Устройства"),
                ("Язык"),
                ("Автоматизация"),
                ("Миниатюры"),
                ("Часовые пояса"),
                ("Обработка ошибок"),
                ("Предупреждения"),
                ("Объединение"),
                ("Разное"),
            )
            icons = (
                "prefs/bw/devices.svg",
                "prefs/bw/language.svg",
                "prefs/bw/automation.svg",
                "prefs/bw/thumbnails.svg",
                "prefs/bw/timezone.svg",
                "prefs/bw/error-handling.svg",
                "prefs/bw/warnings.svg",
                "prefs/bw/consolidation.svg",
                "prefs/bw/miscellaneous.svg",
            )
        else:
            self.chooser_items = (
                ("Устройства"),
                ("Язык"),
                ("Автоматизация"),
                ("Миниатюры"),
                ("Часовые пояса"),
                ("Обработка ошибок"),
                ("Предупреждения"),
                ("Разное"),
            )
            icons = (
                "prefs/bw/devices.svg",
                "prefs/bw/language.svg",
                "prefs/bw/automation.svg",
                "prefs/bw/thumbnails.svg",
                "prefs/bw/timezone.svg",
                "prefs/bw/error-handling.svg",
                "prefs/bw/warnings.svg",
                "prefs/bw/miscellaneous.svg",
            )

        for prefIcon, label in zip(icons, self.chooser_items):
            # make the selected icons be the same colour as the selected text
            icon = QIcon()
            pixmap = QPixmap(data_file_path(prefIcon))
            selected = QPixmap(pixmap.size())
            selected.fill(selectedColour)
            selected.setMask(pixmap.createMaskFromColor(Qt.transparent))
            pixmap = darkModePixmap(pixmap=pixmap)
            icon.addPixmap(pixmap, QIcon.Normal)
            icon.addPixmap(selected, QIcon.Selected)

            item = QListWidgetItem(icon, label, self.chooser)
            item.setFont(QFont())
            width = fontMetrics.width(label) + icon_width + icon_padding * 2
            item.setSizeHint(QSize(width, icon_height * 2))

        self.chooser.currentRowChanged.connect(self.rowChanged)
        self.chooser.setSelectionMode(QAbstractItemView.SingleSelection)
        self.chooser.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.MinimumExpanding)

        self.devices = QWidget()

        self.scanBox = QGroupBox(("Сканирование устройств"))
        self.onlyExternal = QCheckBox(("Сканировать только внешние устройства"))
        self.onlyExternal.setToolTip(
            (
               "Сканировать фото и видео только на внешних устройствах по отношению к этому "
               "компьютеру.\n"
               "Включая камеры, карты памяти, внешние жесткие диски и USB флеш диски."
            )
        )
        self.scanSpecificFolders = QCheckBox(("Сканировать на устройстве только определенные папки"))
        tip = (
            "Сканировать на наличие фото и видео только папки, указанные ниже (за\n"
            "исключением папок заданных в Путях для исключения).\n\n"
            "Изменение этой настройки приведет к повторному сканированию всех устройств."
        )
        self.scanSpecificFolders.setToolTip(tip)

        self.foldersToScanLabel = QLabel(("Папки для сканирования:"))
        self.foldersToScan = QNarrowListWidget(minimum_rows=5)
        self.foldersToScan.setToolTip(
            (
               "Папки в корне файловой системы устройства, которые будут\n"
               "просканированы на наличие фото и видео."
            )
        )
        self.addFolderToScan = QPushButton(("Добавить…"))
        self.addFolderToScan.setToolTip(
            (
               "Добавить папку в список папок для сканирования на наличие фото и видео.\n\n"
               "Изменение этой настройки приведет к повторному сканированию всех устройств."
            )
        )
        self.removeFolderToScan = QPushButton(("Удалить"))
        self.removeFolderToScan.setToolTip(
            (
                "Удалить папку из списка папок для сканирования на наличие фото и видео.\n\n"
                "Изменение этой настройки приведет к повторному сканированию всех устройств."
            )
        )

        self.addFolderToScan.clicked.connect(self.addFolderToScanClicked)
        self.removeFolderToScan.clicked.connect(self.removeFolderToScanClicked)

        scanLayout = QGridLayout()
        scanLayout.setHorizontalSpacing(18)
        scanLayout.addWidget(self.onlyExternal, 0, 0, 1, 3)
        scanLayout.addWidget(self.scanSpecificFolders, 1, 0, 1, 3)
        scanLayout.addWidget(self.foldersToScanLabel, 2, 1, 1, 2)
        scanLayout.addWidget(self.foldersToScan, 3, 1, 3, 1)
        scanLayout.addWidget(self.addFolderToScan, 3, 2, 1, 1)
        scanLayout.addWidget(self.removeFolderToScan, 4, 2, 1, 1)
        self.scanBox.setLayout(scanLayout)

        tip = ("Устройства могут быть настроены для автоматического их игнорирования или автоматической загрузки с них.")
        self.knownDevicesBox = QGroupBox(("Сохраненные устройства"))
        self.knownDevices = QNarrowListWidget(minimum_rows=5)
        self.knownDevices.setToolTip(tip)
        tip = (
            "Удалить устройство из списка устройств для автоматической загрузки или "
            "игнорирования."
        )
        self.removeDevice = QPushButton(("Удалить"))
        self.removeDevice.setToolTip(tip)
        self.removeAllDevice = QPushButton(("Удалить все"))
        tip = (
            "Очистить список устройств для автоматической загрузки или игнорирования.\n\n"
            "Замечание: Изменения вступят в силу при следующем сканировании устройств."
        )
        self.removeAllDevice.setToolTip(tip)
        self.removeDevice.clicked.connect(self.removeDeviceClicked)
        self.removeAllDevice.clicked.connect(self.removeAllDeviceClicked)
        knownDevicesLayout = QGridLayout()
        knownDevicesLayout.setHorizontalSpacing(18)
        knownDevicesLayout.addWidget(self.knownDevices, 0, 0, 3, 1)
        knownDevicesLayout.addWidget(self.removeDevice, 0, 1, 1, 1)
        knownDevicesLayout.addWidget(self.removeAllDevice, 1, 1, 1, 1)
        self.knownDevicesBox.setLayout(knownDevicesLayout)

        self.ignoredPathsBox = QGroupBox(("Игнорируемые пути"))
        tip = (
            "Конечная часть путей, которые никогда не будут сканироваться на наличие фото или видео."
        )
        self.ignoredPaths = QNarrowListWidget(minimum_rows=4)
        self.ignoredPaths.setToolTip(tip)
        self.addPath = QPushButton(("Добавить…"))
        self.addPath.setToolTip(
            (
               "Добавить путь в список путей для игнорирования.\n\n"
               "Изменение этого параметра приведет к повторному сканированию всех устройств."
            )
        )
        self.removePath = QPushButton(("Удалить"))
        self.removePath.setToolTip(
            (
                "Удалить путь из списка путей для игнорирования.\n\n"
                "Изменение этого параметра приведет к повторному сканированию всех устройств."
            )
        )
        self.removeAllPath = QPushButton(("Удалить все"))
        self.removeAllPath.setToolTip(
            (
                "Очистить список путей для игнорирования.\n\n"
                "Изменение этого параметра приведет к повторному сканированию всех устройств."
            )
        )
        self.addPath.clicked.connect(self.addPathClicked)
        self.removePath.clicked.connect(self.removePathClicked)
        self.removeAllPath.clicked.connect(self.removeAllPathClicked)
        self.ignoredPathsRe = QCheckBox()
        self.ignorePathsReLabel = ClickableLabel(
            ("Использовать <a {link}>регулярные выражения</a> в стиле Python").format(
                link='style="text-decoration:none; color: palette(highlight);"'
                'href="http://sdr-technology.ru"'
            )
        )
        self.ignorePathsReLabel.setToolTip(
            (
               "Использовать регулярные выражения в списке игнорируемых путей.\n\n"
               "Изменение этого параметра приведет к повторному сканированию всех устройств."
            )
        )
        self.ignorePathsReLabel.setTextInteractionFlags(Qt.TextBrowserInteraction)
        self.ignorePathsReLabel.setOpenExternalLinks(True)
        self.ignorePathsReLabel.clicked.connect(self.ignorePathsReLabelClicked)
        reLayout = QHBoxLayout()
        reLayout.setSpacing(5)
        reLayout.addWidget(self.ignoredPathsRe)
        reLayout.addWidget(self.ignorePathsReLabel)
        reLayout.addStretch()
        ignoredPathsLayout = QGridLayout()
        ignoredPathsLayout.setHorizontalSpacing(18)
        ignoredPathsLayout.addWidget(self.ignoredPaths, 0, 0, 4, 1)
        ignoredPathsLayout.addWidget(self.addPath, 0, 1, 1, 1)
        ignoredPathsLayout.addWidget(self.removePath, 1, 1, 1, 1)
        ignoredPathsLayout.addWidget(self.removeAllPath, 2, 1, 1, 1)
        ignoredPathsLayout.addLayout(reLayout, 4, 0, 1, 2)
        self.ignoredPathsBox.setLayout(ignoredPathsLayout)

        self.setDeviceWidgetValues()

        # connect these next 3 only after having set their values, so rescan / search
        # again in sdrApp is not triggered
        self.onlyExternal.stateChanged.connect(self.onlyExternalChanged)
        self.scanSpecificFolders.stateChanged.connect(self.noDcimChanged)
        self.ignoredPathsRe.stateChanged.connect(self.ignoredPathsReChanged)

        devicesLayout = QVBoxLayout()
        devicesLayout.addWidget(self.scanBox)
        devicesLayout.addWidget(self.ignoredPathsBox)
        devicesLayout.addWidget(self.knownDevicesBox)
        devicesLayout.addStretch()
        devicesLayout.setSpacing(18)

        self.devices.setLayout(devicesLayout)
        devicesLayout.setContentsMargins(0, 0, 0, 0)

        self.language = QWidget()
        self.languages = QComboBox()
        self.languages.setEditable(False)
        self.languagesLabel = QLabel(("Язык: "))
        self.languages.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        # self.languages.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)

        self.setLanguageWidgetValues()

        self.languages.currentIndexChanged.connect(self.languagesChanged)

        languageWidgetsLayout = QHBoxLayout()
        languageWidgetsLayout.addWidget(self.languagesLabel)
        languageWidgetsLayout.addWidget(self.languages)
        languageWidgetsLayout.addWidget(QLabel(("*")))
        languageWidgetsLayout.addStretch()
        languageWidgetsLayout.setSpacing(5)

        languageLayout = QVBoxLayout()
        languageLayout.addLayout(languageWidgetsLayout)
        languageLayout.addWidget(QLabel(("* Вступит в силу после перезапуска программы")))
        languageLayout.addStretch()
        languageLayout.setContentsMargins(0, 0, 0, 0)
        languageLayout.setSpacing(18)
        self.language.setLayout(languageLayout)

        self.automation = QWidget()

        self.automationBox = QGroupBox(("Автоматизация программы"))
        self.autoMount = QCheckBox(("Подключаемые устройства не подключены автоматически"))
        tooltip = (
             "Подключаемые устройства такие как карты памяти или внешние диски\n"
             "когда операционная система не подключает их автоматически"
        )
        self.autoMount.setToolTip(tooltip)
        self.autoDownloadStartup = QCheckBox(("Начать загрузку при запуске программы"))
        self.autoDownloadInsertion = QCheckBox(
            ("Начать загрузку сразу после подключения носителя")
        )
        self.autoEject = QCheckBox(("Отмонтировать (извлечь) устройство по окончании загрузки"))
        self.autoExit = QCheckBox(("Завершить программу после загрузки"))
        self.autoExitError = QCheckBox(
            ("Завершить программу даже при ошибках загрузки")
        )
        self.setAutomationWidgetValues()
        self.autoMount.stateChanged.connect(self.autoMountChanged)
        self.autoDownloadStartup.stateChanged.connect(self.autoDownloadStartupChanged)
        self.autoDownloadInsertion.stateChanged.connect(
            self.autoDownloadInsertionChanged
        )
        self.autoEject.stateChanged.connect(self.autoEjectChanged)
        self.autoExit.stateChanged.connect(self.autoExitChanged)
        self.autoExitError.stateChanged.connect(self.autoExitErrorChanged)

        automationBoxLayout = QGridLayout()
        automationBoxLayout.addWidget(self.autoMount, 0, 0, 1, 2)
        automationBoxLayout.addWidget(self.autoDownloadStartup, 1, 0, 1, 2)
        automationBoxLayout.addWidget(self.autoDownloadInsertion, 2, 0, 1, 2)
        automationBoxLayout.addWidget(self.autoEject, 3, 0, 1, 2)
        automationBoxLayout.addWidget(self.autoExit, 4, 0, 1, 2)
        automationBoxLayout.addWidget(self.autoExitError, 5, 1, 1, 1)
        checkbox_width = self.autoExit.style().pixelMetric(QStyle.PM_IndicatorWidth)
        automationBoxLayout.setColumnMinimumWidth(0, checkbox_width)
        self.automationBox.setLayout(automationBoxLayout)

        automationLayout = QVBoxLayout()
        automationLayout.addWidget(self.automationBox)
        automationLayout.addStretch()
        automationLayout.setContentsMargins(0, 0, 0, 0)

        self.automation.setLayout(automationLayout)

        self.performance = QWidget()

        self.performanceBox = QGroupBox(("Создание миниатюр"))
        self.generateThumbnails = QCheckBox(("Создать миниатюры"))
        self.generateThumbnails.setToolTip(
            ("Создать миниатюры для отображения в главном окне программы")
        )
        self.useThumbnailCache = QCheckBox(("Кэшировать миниатюры"))
        self.useThumbnailCache.setToolTip(
            (
               "Сохранять отображаемые в главном окне программы миниатюры в персональном "
               "кеше миниатюр SmartVision Guard"
            )
        )
        self.fdoThumbnails = QCheckBox(("Создать миниатюры в системе"))
        self.fdoThumbnails.setToolTip(
            (
                 "Во время загрузки сохранять миниатюры, которые могут быть использованы файл "
                 "менеджерами и другими программами"
            )
        )
        self.generateThumbnails.stateChanged.connect(self.generateThumbnailsChanged)
        self.useThumbnailCache.stateChanged.connect(self.useThumbnailCacheChanged)
        self.fdoThumbnails.stateChanged.connect(self.fdoThumbnailsChanged)
        self.maxCores = QComboBox()
        self.maxCores.setEditable(False)
        tip = ("Количество ядер процессора для использования при создании миниатюр.")
        self.coresLabel = QLabel(("Ядра процессора:"))
        self.coresLabel.setToolTip(tip)
        self.maxCores.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.maxCores.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        self.maxCores.setToolTip(tip)

        self.setPerformanceValues()

        self.maxCores.currentIndexChanged.connect(self.maxCoresChanged)

        coresLayout = QHBoxLayout()
        coresLayout.addWidget(self.coresLabel)
        coresLayout.addWidget(self.maxCores)
        coresLayout.addWidget(QLabel(("*")))
        coresLayout.addStretch()

        performanceBoxLayout = QVBoxLayout()
        performanceBoxLayout.addWidget(self.generateThumbnails)
        performanceBoxLayout.addWidget(self.useThumbnailCache)
        performanceBoxLayout.addWidget(self.fdoThumbnails)
        performanceBoxLayout.addLayout(coresLayout)
        self.performanceBox.setLayout(performanceBoxLayout)

        self.thumbnail_cache = ThumbnailCacheSql(create_table_if_not_exists=False)

        self.cacheSize = CacheSize()
        self.cacheSizeThread = QThread()
        self.cacheSizeThread.started.connect(self.cacheSize.start)
        self.getCacheSize.connect(self.cacheSize.getCacheSize)
        self.cacheSize.size.connect(self.setCacheSize)
        self.cacheSize.moveToThread(self.cacheSizeThread)

        QTimer.singleShot(0, self.cacheSizeThread.start)

        self.getCacheSize.emit()

        self.cacheBox = QGroupBox(("Кэш миниатюр"))
        self.thumbnailCacheSize = QLabel()
        self.thumbnailCacheSize.setText(("Вычисление…"))
        self.thumbnailNumber = QLabel()
        self.thumbnailSqlSize = QLabel()
        self.thumbnailCacheDaysKeep = QSpinBox()
        self.thumbnailCacheDaysKeep.setMinimum(0)
        self.thumbnailCacheDaysKeep.setMaximum(360 * 3)
        self.thumbnailCacheDaysKeep.setSuffix(" " + ("день(дня,дней)"))
        self.thumbnailCacheDaysKeep.setSpecialValueText(("всегда"))
        self.thumbnailCacheDaysKeep.valueChanged.connect(
            self.thumbnailCacheDaysKeepChanged
        )

        cacheBoxLayout = QVBoxLayout()
        cacheLayout = QGridLayout()
        cacheLayout.addWidget(QLabel(("Размер кэша:")), 0, 0, 1, 1)
        cacheLayout.addWidget(self.thumbnailCacheSize, 0, 1, 1, 1)
        cacheLayout.addWidget(QLabel(("Количество миниатюр:")), 1, 0, 1, 1)
        cacheLayout.addWidget(self.thumbnailNumber, 1, 1, 1, 1)
        cacheLayout.addWidget(QLabel(("Размер базы данных:")), 2, 0, 1, 1)
        cacheLayout.addWidget(self.thumbnailSqlSize, 2, 1, 1, 1)
        cacheLayout.addWidget(QLabel(("В кэше недоступны миниатюры для:")), 3, 0, 1, 1)
        cacheDays = QHBoxLayout()
        cacheDays.addWidget(self.thumbnailCacheDaysKeep)
        cacheDays.addWidget(QLabel(("*")))
        cacheLayout.addLayout(cacheDays, 3, 1, 1, 1)
        cacheBoxLayout.addLayout(cacheLayout)

        cacheButtons = QDialogButtonBox()
        self.purgeCache = cacheButtons.addButton(
            ("Очистка кэша..."), QDialogButtonBox.ResetRole
        )
        self.optimizeCache = cacheButtons.addButton(
            ("Оптимизация кэша..."), QDialogButtonBox.ResetRole
        )
        self.purgeCache.clicked.connect(self.purgeCacheClicked)
        self.optimizeCache.clicked.connect(self.optimizeCacheClicked)

        cacheBoxLayout.addWidget(cacheButtons)

        self.cacheBox.setLayout(cacheBoxLayout)
        self.setCacheValues()

        performanceLayout = QVBoxLayout()
        performanceLayout.addWidget(self.performanceBox)
        performanceLayout.addWidget(self.cacheBox)
        performanceLayout.addWidget(QLabel(("* Вступит в силу после перезапуска программы")))
        performanceLayout.addStretch()
        performanceLayout.setContentsMargins(0, 0, 0, 0)
        performanceLayout.setSpacing(18)

        self.performance.setLayout(performanceLayout)

        self.timeZone = QWidget()

        self.timeZoneBox = QGroupBox(("Часовые пояса"))
        self.ignoreTimeZone = QCheckBox(
            ("Игнорировать изменения часового пояса и зимнего времени")
        )
        self.timeZoneOffsetResolution = QComboBox()
        self.timeZoneOffsetResolution.setEditable(False)
        self.timeZoneOffsetResolution.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.timeZoneOffsetResolution.setSizePolicy(
            QSizePolicy.Minimum, QSizePolicy.Minimum
        )
        self.timeZoneOffsetResolution.addItems(("60", "30", "15"))
        self.timeZoneOffsetLabel = QLabel(("Шаг смещения (минут)"))
        tooltip = (
            "При вычислении смещения относительно времени когда сделано фото или видео "
            "используется несколько"
        )
        self.timeZoneOffsetLabel.setToolTip(tooltip)
        self.timeZoneOffsetResolution.setToolTip(tooltip)
        self.timeZoneOffset = QWidget()
        timeZoneOffsetLayout = QHBoxLayout()
        timeZoneOffsetLayout.addWidget(self.timeZoneOffsetResolution)
        timeZoneOffsetLayout.addStretch()
        timeZoneOffsetLayout.setContentsMargins(0, 0, 0, 0)
        self.timeZoneOffset.setLayout(timeZoneOffsetLayout)

        timeZoneExplanation = QLabel(
            ("При определении, был ли файл уже загружен:")
        )

        timeZoneBoxLayout = QGridLayout()
        timeZoneBoxLayout.addWidget(timeZoneExplanation, 0, 0, 1, 3)
        timeZoneBoxLayout.addWidget(self.ignoreTimeZone, 1, 0, 1, 3)
        timeZoneBoxLayout.addWidget(self.timeZoneOffsetLabel, 2, 1, 1, 1)
        timeZoneBoxLayout.addWidget(self.timeZoneOffset, 2, 2, 1, 1)
        timeZoneBoxLayout.setColumnMinimumWidth(0, checkbox_width)
        self.timeZoneBox.setLayout(timeZoneBoxLayout)

        timeZoneHelpLink = StyledLinkLabel()
        timeZoneHelpLink.setLink(
            url="https://yandex.ru",
            text=("Узнать больше о работе с часовыми поясами"),
        )
        timeZoneHelpLink.setWordWrap(True)
        timeZoneHelpLink.setOpenExternalLinks(True)

        timeZoneLayout = QVBoxLayout()
        timeZoneLayout.addWidget(self.timeZoneBox)
        timeZoneLayout.addWidget(timeZoneHelpLink)
        timeZoneLayout.addStretch()
        timeZoneLayout.setContentsMargins(0, 0, 0, 0)
        timeZoneLayout.setSpacing(18)

        self.timeZone.setLayout(timeZoneLayout)

        self.setTimeZoneValues()
        self.ignoreTimeZone.stateChanged.connect(self.ignoreTimeZoneChanged)
        self.timeZoneOffsetResolution.currentIndexChanged.connect(
            self.timeZoneOffsetResolutionChanged
        )

        self.errorBox = QGroupBox(("Обработка ошибок"))

        self.downloadErrorGroup = QButtonGroup()
        self.skipDownload = QRadioButton(("Пропустить загрузку"))
        self.skipDownload.setToolTip(
            ("Не загружать файл, и выдать сообщение об ошибке")
        )
        self.addIdentifier = QRadioButton(("Добавить уникальную метку"))
        self.addIdentifier.setToolTip(
            (
                "Добавить метку вида  _1 или _2 в конец имени файла, непосредственно перед "
                "расширением"
            )
        )
        self.downloadErrorGroup.addButton(self.skipDownload)
        self.downloadErrorGroup.addButton(self.addIdentifier)

        self.backupErrorGroup = QButtonGroup()
        self.overwriteBackup = QRadioButton(("Перезаписать"))
        self.overwriteBackup.setToolTip(("Перезаписать предыдущую резервную копию файлаe"))
        self.skipBackup = QRadioButton(("Пропустить"))
        self.skipBackup.setToolTip(
            ("Не перезаписывать резервную копию, выдать сообщение об ошибке")
        )
        self.backupErrorGroup.addButton(self.overwriteBackup)
        self.backupErrorGroup.addButton(self.skipBackup)

        errorBoxLayout = QVBoxLayout()
        lbl = (
            "Если фото или видео с таким именем уже было загружено, выберите, пропустить "
            "ли загрузку файла или добавить уникальную метку:"
        )
        self.downloadError = QLabel(lbl)
        self.downloadError.setWordWrap(True)
        errorBoxLayout.addWidget(self.downloadError)
        errorBoxLayout.addWidget(self.skipDownload)
        errorBoxLayout.addWidget(self.addIdentifier)
        lbl = (
            "<i>"
            + (
                "Для автоматического создания уникальных имен файлов настоятельно "
                "рекомендуется использовать нумерацию. Настройте переименование файлов на "
                "панели Переименование в главном окне."
            )
            + "</i>"
        )
        self.recommended = QLabel(lbl)
        self.recommended.setWordWrap(True)
        errorBoxLayout.addWidget(self.recommended)
        errorBoxLayout.addSpacing(18)
        lbl = (
                "При создании резервной копии, выберите, перезаписывать ли файл с таким "
                "именем на устройстве для резервного копирования или пропустить создание "
                "резервной копии:"
        )
        self.backupError = QLabel(lbl)
        self.backupError.setWordWrap(True)
        errorBoxLayout.addWidget(self.backupError)
        errorBoxLayout.addWidget(self.overwriteBackup)
        errorBoxLayout.addWidget(self.skipBackup)
        self.errorBox.setLayout(errorBoxLayout)

        self.setErrorHandingValues()
        self.downloadErrorGroup.buttonClicked.connect(self.downloadErrorGroupClicked)
        self.backupErrorGroup.buttonClicked.connect(self.backupErrorGroupClicked)

        self.errorWidget = QWidget()
        errorLayout = QVBoxLayout()
        self.errorWidget.setLayout(errorLayout)
        errorLayout.addWidget(self.errorBox)
        errorLayout.addStretch()
        errorLayout.setContentsMargins(0, 0, 0, 0)

        self.warningBox = QGroupBox(("Предупреждения программы"))
        lbl = ("Показывать предупреждение когда:")
        self.warningLabel = QLabel(lbl)
        self.warningLabel.setWordWrap(True)
        self.warnDownloadingAll = QCheckBox(
            ("Загружаемые файлы сейчас не отображаются")
        )
        tip = (
            "Предупреждать о загрузке файлов, которые не отображаются в главном окне."
        )
        self.warnDownloadingAll.setToolTip(tip)
        self.warnBackupProblem = QCheckBox(("Места хранения резервных копий отсутствуют"))
        tip = (
            "Предупреждать перед началом загрузки, если создание резервных копий невозможно."
        )
        self.warnBackupProblem.setToolTip(tip)
        self.warnMissingLibraries = QCheckBox(
            ("Библиотеки программы отсутствуют или повреждены")
        )
        tip = (
            "Предупреждать, если библиотеки, используемые SmartVision Guard отсутствуют "
            "или не работают."
        )
        self.warnMissingLibraries.setToolTip(tip)
        self.warnMetadata = QCheckBox(("Метаданные файловой системы не могут быть установлены"))
        tip = (
            "Предупреждать об ошибке установки метаданных файловой системы, таких как "
            "время изменения."
        )
        self.warnMetadata.setToolTip(tip)
        self.warnUnhandledFiles = QCheckBox(("Найдены необслуживаемые файлы"))
        tip = (
            "Предупреждать после сканирования устройств или этого компьютера, если есть "
            "неопознанные файлы, которые не будет включены в загрузку."
        )
        self.warnUnhandledFiles.setToolTip(tip)
        self.exceptTheseFilesLabel = QLabel(
            ("Не сообщать о необслуживаемых файлах с расширениями:")
        )
        self.exceptTheseFilesLabel.setWordWrap(True)
        self.exceptTheseFiles = QNarrowListWidget(minimum_rows=4)
        tip = (
            "Расширения файлов не чувствительны к регистру и не обязаны содержать "
            "предшествующую точку."
        )
        self.exceptTheseFiles.setToolTip(tip)
        self.addExceptFiles = QPushButton(("Добавить"))
        tip = (
            "Добавить расширение файла в список необслуживаемых типов файлов, сообщать о "
            "которых не нужно."
        )
        self.addExceptFiles.setToolTip(tip)
        tip = (
            "Удалить расширение файла из списка необслуживаемых типов файлов, сообщать о "
            "которых не нужно."
        )
        self.removeExceptFiles = QPushButton(("Удалить"))
        self.removeExceptFiles.setToolTip(tip)
        self.removeAllExceptFiles = QPushButton(("Удалить все"))
        tip = (
            "Очистить список необслуживаемых типов файлов, сообщать о которых не нужно."
        )
        self.removeAllExceptFiles.setToolTip(tip)
        self.addExceptFiles.clicked.connect(self.addExceptFilesClicked)
        self.removeExceptFiles.clicked.connect(self.removeExceptFilesClicked)
        self.removeAllExceptFiles.clicked.connect(self.removeAllExceptFilesClicked)

        self.setWarningValues()
        self.warnDownloadingAll.stateChanged.connect(self.warnDownloadingAllChanged)
        self.warnBackupProblem.stateChanged.connect(self.warnBackupProblemChanged)
        self.warnMissingLibraries.stateChanged.connect(self.warnMissingLibrariesChanged)
        self.warnMetadata.stateChanged.connect(self.warnMetadataChanged)
        self.warnUnhandledFiles.stateChanged.connect(self.warnUnhandledFilesChanged)

        warningBoxLayout = QGridLayout()
        warningBoxLayout.addWidget(self.warningLabel, 0, 0, 1, 3)
        warningBoxLayout.addWidget(self.warnDownloadingAll, 1, 0, 1, 3)
        warningBoxLayout.addWidget(self.warnBackupProblem, 2, 0, 1, 3)
        warningBoxLayout.addWidget(self.warnMissingLibraries, 3, 0, 1, 3)
        warningBoxLayout.addWidget(self.warnMetadata, 4, 0, 1, 3)
        warningBoxLayout.addWidget(self.warnUnhandledFiles, 5, 0, 1, 3)
        warningBoxLayout.addWidget(self.exceptTheseFilesLabel, 6, 1, 1, 2)
        warningBoxLayout.addWidget(self.exceptTheseFiles, 7, 1, 4, 1)
        warningBoxLayout.addWidget(self.addExceptFiles, 7, 2, 1, 1)
        warningBoxLayout.addWidget(self.removeExceptFiles, 8, 2, 1, 1)
        warningBoxLayout.addWidget(self.removeAllExceptFiles, 9, 2, 1, 1)
        warningBoxLayout.setColumnMinimumWidth(0, checkbox_width)
        self.warningBox.setLayout(warningBoxLayout)

        self.warnings = QWidget()
        warningLayout = QVBoxLayout()
        self.warnings.setLayout(warningLayout)
        warningLayout.addWidget(self.warningBox)
        warningLayout.addStretch()
        warningLayout.setContentsMargins(0, 0, 0, 0)

        if CONSOLIDATION_IMPLEMENTED:
            self.consolidationBox = QGroupBox(("Объединение фото и видео"))

            self.consolidateIdentical = QCheckBox(
                ("Объединять файлы между устройствами и загрузками")
            )
            tip = (
                "Анализировать результаты сканирования устройств в поисках дубликатов и пар "
                "RAW и JPEG,\n"
                "сравнивая их между различными устройствами и сеансами загрузки."
            )
            self.consolidateIdentical.setToolTip(tip)

            self.treatRawJpegLabel = QLabel(("Рассматривать соответствующие RAW и JPEG файлы как:"))
            self.oneRawJpeg = QRadioButton(("Одно фото"))
            self.twoRawJpeg = QRadioButton(("Два фото"))
            tip = (
                "Отображать пары RAW и JPEG как одно фото, и, если отмечены для загрузки, "
                "загружать оба файла."
            )
            self.oneRawJpeg.setToolTip(tip)
            tip = (
                "Отображать пары RAW и JPEG как два разных фото. Вы по-прежнему можете "
                "синхронизировать их нумерацию."
            )
            self.twoRawJpeg.setToolTip(tip)

            self.treatRawJpegGroup = QButtonGroup()
            self.treatRawJpegGroup.addButton(self.oneRawJpeg)
            self.treatRawJpegGroup.addButton(self.twoRawJpeg)

            self.markRawJpegLabel = QLabel(("С парами RAW и JPEG фото:"))

            self.noJpegWhenRaw = QRadioButton(("Не отмечать JPEG для загрузки"))
            self.noRawWhenJpeg = QRadioButton(("Не отмечать RAW для загрузки"))
            self.markRawJpeg = QRadioButton(("Отмечать для загрузки оба"))

            self.markRawJpegGroup = QButtonGroup()
            for widget in (self.noJpegWhenRaw, self.noRawWhenJpeg, self.markRawJpeg):
                self.markRawJpegGroup.addButton(widget)

            tip = (
                "При обнаружении пар RAW и JPEG фото, не отмечать автоматически JPEG файлы\n"
                "для загрузки. Вы по-прежнему можете отметить их для загрузки сами."
            )
            self.noJpegWhenRaw.setToolTip(tip)
            tip = (
                "При обнаружении пар RAW и JPEG фото, не отмечать автоматически RAW файлы\n"
                "для загрузки. Вы по-прежнему можете отметить их для загрузки сами."
            )
            self.noRawWhenJpeg.setToolTip(tip)
            tip = (
                "При обнаружении пар RAW и JPEG фото, автоматически отмечать для загрузки оба "
                "файла."
            )
            self.markRawJpeg.setToolTip(tip)

            explanation = (
                "Если вы отключили объединение файлов, выберите, что делать если подключено "
                "устройство для загрузки в то время как отображаются завершенные загрузки:"
            )

        else:
            explanation = (
                "Если устройство для загрузки подключено в то время как отображаются "
                "завершенные загрузки:"
            )
        self.noconsolidationLabel = QLabel(explanation)
        self.noconsolidationLabel.setWordWrap(True)
        self.noconsolidationLabel.setSizePolicy(
            QSizePolicy.Ignored, QSizePolicy.Minimum
        )
        # Unless this next call is made, for some reason the widget is far too high! :-(
        self.noconsolidationLabel.setContentsMargins(0, 0, 1, 0)

        self.noConsolidationGroup = QButtonGroup()
        self.noConsolidationGroup.buttonClicked.connect(
            self.noConsolidationGroupClicked
        )

        self.clearCompletedDownloads = QRadioButton(("Очистить завершенные загрузки"))
        self.keepCompletedDownloads = QRadioButton(
            ("Продолжить отображение завершенных загрузок")
        )
        self.promptCompletedDownloads = QRadioButton(("Спросить что делать"))
        self.noConsolidationGroup.addButton(self.clearCompletedDownloads)
        self.noConsolidationGroup.addButton(self.keepCompletedDownloads)
        self.noConsolidationGroup.addButton(self.promptCompletedDownloads)
        tip = (
            "Автоматически очищать отображаемые завершенные загрузки каждый раз, когда "
            "подключается новое устройство для загрузки."
        )
        self.clearCompletedDownloads.setToolTip(tip)
        tip = (
            "Продолжать отображать завершенные загрузки каждый раз, когда подключается "
            "новое устройство для загрузки."
        )
        self.keepCompletedDownloads.setToolTip(tip)
        tip = (
            "Спросить, очищать отображаемые завершенные загрузки или продолжать их "
            "отображение каждый раз, когда подключается новое устройство для загрузки."
        )
        self.promptCompletedDownloads.setToolTip(tip)

        if CONSOLIDATION_IMPLEMENTED:
            consolidationBoxLayout = QGridLayout()
            consolidationBoxLayout.addWidget(self.consolidateIdentical, 0, 0, 1, 3)

            consolidationBoxLayout.addWidget(self.treatRawJpegLabel, 1, 1, 1, 2)
            consolidationBoxLayout.addWidget(self.oneRawJpeg, 2, 1, 1, 2)
            consolidationBoxLayout.addWidget(self.twoRawJpeg, 3, 1, 1, 2)

            consolidationBoxLayout.addWidget(self.markRawJpegLabel, 4, 2, 1, 1)
            consolidationBoxLayout.addWidget(self.noJpegWhenRaw, 5, 2, 1, 1)
            consolidationBoxLayout.addWidget(self.noRawWhenJpeg, 6, 2, 1, 1)
            consolidationBoxLayout.addWidget(self.markRawJpeg, 7, 2, 1, 1, Qt.AlignTop)

            consolidationBoxLayout.addWidget(self.noconsolidationLabel, 8, 0, 1, 3)
            consolidationBoxLayout.addWidget(self.keepCompletedDownloads, 9, 0, 1, 3)
            consolidationBoxLayout.addWidget(self.clearCompletedDownloads, 10, 0, 1, 3)
            consolidationBoxLayout.addWidget(self.promptCompletedDownloads, 11, 0, 1, 3)

            consolidationBoxLayout.setColumnMinimumWidth(0, checkbox_width)
            consolidationBoxLayout.setColumnMinimumWidth(1, checkbox_width)

            consolidationBoxLayout.setRowMinimumHeight(7, checkbox_width * 2)

            self.consolidationBox.setLayout(consolidationBoxLayout)

            self.consolidation = QWidget()
            consolidationLayout = QVBoxLayout()
            consolidationLayout.addWidget(self.consolidationBox)
            consolidationLayout.addStretch()
            consolidationLayout.setContentsMargins(0, 0, 0, 0)
            consolidationLayout.setSpacing(18)
            self.consolidation.setLayout(consolidationLayout)

            self.setCompletedDownloadsValues()
            self.setConsolidatedValues()
            self.consolidateIdentical.stateChanged.connect(
                self.consolidateIdenticalChanged
            )
            self.treatRawJpegGroup.buttonClicked.connect(self.treatRawJpegGroupClicked)
            self.markRawJpegGroup.buttonClicked.connect(self.markRawJpegGroupClicked)

        self.metadataBox = QGroupBox(("Метаданные"))
        self.ignoreMdatatimeMtpDng = QCheckBox(
            ("Игнорировать метаданные даты/времени DNG файлов на устройствах MTP")
        )
        tip = (
            "Игнорировать метаданные даты/времени DNG файлов, расположенных на устройствах "
            "MTP, и использовать вместо них время изменения.\n\n"
            "Полезно для таких устройств, как некоторые телефоны и планшеты, которые "
            "создают некорректные метаданные DNG."
        )
        self.ignoreMdatatimeMtpDng.setToolTip(tip)

        self.forceExiftool = QCheckBox(("Считывать метаданные снимков используя только ExifTool"))
        tip = (
            "Для чтения метаданных и извлечения миниатюр использовать ExifTool вместо Exiv2.\n\n"
            "По умолчанию используется Exiv2, полагаясь на ExifTool только когда Exiv2 не\n"
            "поддерживает формат считываемого файла.\n\n"
            "Exiv2 быстр, точен, и почти всегда надежен, но он не справляется с извлечением\n"
            "метаданных из некоторых файлов, таких как DNG файлы"
        )

        self.forceExiftool.setToolTip(tip)

        self.forceExiftoolVideo = QCheckBox(
            ("Считывать метаданные видео используя только ExifTool")
        )
        tip = (
            "<p>Использовать ExifTool вместо MediaInfo и Gstreamer для считывания "
            "метаданнных видео и извлечения превью.</p><p>по умолчанию используется "
            "MediaInfo и Gstreamer, с тремя исключениями:</p><ol><li>ExifTool сообщает "
            "часовой пояс, а MediaInfo нет.</li><li>ExifTool предоставляет более надежные "
            "данные, чем MediaInfo для некоторых форматов файлов.</li><li>Gstreamer не "
            "может извлекать превью.</li></ol>"
        )
        self.forceExiftoolVideo.setToolTip(tip)

        self.setMetdataValues()
        self.ignoreMdatatimeMtpDng.stateChanged.connect(
            self.ignoreMdatatimeMtpDngChanged
        )
        self.forceExiftool.stateChanged.connect(self.forceExiftoolChanged)
        self.forceExiftoolVideo.stateChanged.connect(self.forceExiftoolVideoChanged)

        metadataLayout = QVBoxLayout()
        metadataLayout.addWidget(self.ignoreMdatatimeMtpDng)
        metadataLayout.addWidget(self.forceExiftool)
        if FORCE_EXIFTOOL_VIDEO_IMPLEMENTED:
            metadataLayout.addWidget(self.forceExiftoolVideo)
        self.metadataBox.setLayout(metadataLayout)

        if not CONSOLIDATION_IMPLEMENTED:
            self.completedDownloadsBox = QGroupBox(("Завершенные загрузки"))
            completedDownloadsLayout = QVBoxLayout()
            completedDownloadsLayout.addWidget(self.noconsolidationLabel)
            completedDownloadsLayout.addWidget(self.keepCompletedDownloads)
            completedDownloadsLayout.addWidget(self.clearCompletedDownloads)
            completedDownloadsLayout.addWidget(self.promptCompletedDownloads)
            self.completedDownloadsBox.setLayout(completedDownloadsLayout)
            self.setCompletedDownloadsValues()

        self.miscWidget = QWidget()
        miscLayout = QVBoxLayout()
        miscLayout.addWidget(self.metadataBox)
        if not CONSOLIDATION_IMPLEMENTED:
            miscLayout.addWidget(self.completedDownloadsBox)
        miscLayout.addStretch()
        miscLayout.setContentsMargins(0, 0, 0, 0)
        miscLayout.setSpacing(18)
        self.miscWidget.setLayout(miscLayout)

        self.panels.addWidget(self.devices)
        self.panels.addWidget(self.language)
        self.panels.addWidget(self.automation)
        self.panels.addWidget(self.performance)
        self.panels.addWidget(self.timeZone)
        self.panels.addWidget(self.errorWidget)
        self.panels.addWidget(self.warnings)
        if CONSOLIDATION_IMPLEMENTED:
            self.panels.addWidget(self.consolidation)
        self.panels.addWidget(self.miscWidget)

        layout = QVBoxLayout()
        self.setLayout(layout)
        layout.setSpacing(layout.contentsMargins().left() * 2)
        layout.setContentsMargins(18, 18, 18, 18)

        buttons = QDialogButtonBox(
            QDialogButtonBox.RestoreDefaults
            | QDialogButtonBox.Close
            | QDialogButtonBox.Help
        )
        translateDialogBoxButtons(buttons)
        self.restoreButton: QPushButton = buttons.button(
            QDialogButtonBox.RestoreDefaults
        )
        self.restoreButton.clicked.connect(self.restoreDefaultsClicked)
        self.helpButton: QPushButton = buttons.button(QDialogButtonBox.Help)
        self.helpButton.clicked.connect(self.helpButtonClicked)
        self.helpButton.setToolTip(("Помощь онлайн..."))
        self.closeButton: QPushButton = buttons.button(QDialogButtonBox.Close)
        self.closeButton.clicked.connect(self.close)

        controlsLayout = QHBoxLayout()
        controlsLayout.addWidget(self.chooser)
        controlsLayout.addWidget(self.panels)

        controlsLayout.setStretch(0, 0)
        controlsLayout.setStretch(1, 1)
        controlsLayout.setSpacing(layout.contentsMargins().left())

        layout.addLayout(controlsLayout)
        layout.addWidget(buttons)

        self.device_right_side_buttons = (
            self.removeDevice,
            self.removeAllDevice,
            self.addPath,
            self.removePath,
            self.removeAllPath,
        )

        self.device_list_widgets = (self.knownDevices, self.ignoredPaths)
        self.chooser.setCurrentRow(0)

    def reject(self) -> None:
        # If not called, rejecting this dialog will cause SmartVision Guard to
        # crash
        self.close()

    def _addItems(self, pref_list: str, pref_type: int) -> None:
        if self.prefs.list_not_empty(key=pref_list):
            for value in self.prefs[pref_list]:
                QListWidgetItem(value, self.knownDevices, pref_type)

    def setDeviceWidgetValues(self) -> None:
        self.onlyExternal.setChecked(self.prefs.only_external_mounts)
        self.scanSpecificFolders.setChecked(self.prefs.scan_specific_folders)
        self.setFoldersToScanWidgetValues()
        self.knownDevices.clear()
        self._addItems("volume_whitelist", KnownDeviceType.volume_whitelist)
        self._addItems("volume_blacklist", KnownDeviceType.volume_blacklist)
        self._addItems("camera_blacklist", KnownDeviceType.camera_blacklist)
        if self.knownDevices.count():
            self.knownDevices.setCurrentRow(0)
        self.removeDevice.setEnabled(self.knownDevices.count())
        self.removeAllDevice.setEnabled(self.knownDevices.count())
        self.setIgnorePathWidgetValues()

    def setLanguageWidgetValues(self) -> None:
        self.languages.addItem(("<Язык системы>"), system_language)

        filtered = [

            (code, language)

            for code, language in available_languages(display_locale_code=self.prefs.language)

            if (code or '').split('_')[0].lower() in ('en', 'ru')

        ]

        for code, language in filtered:

            self.languages.addItem(language, code)
        value = self.prefs.language
        if value:
            index = self.languages.findData(value)
            self.languages.setCurrentIndex(index)

    def setFoldersToScanWidgetValues(self) -> None:
        self.foldersToScan.clear()
        if self.prefs.list_not_empty("folders_to_scan"):
            self.foldersToScan.addItems(self.prefs.folders_to_scan)
            self.foldersToScan.setCurrentRow(0)
        self.setFoldersToScanState()

    def setFoldersToScanState(self) -> None:
        scan_specific = self.prefs.scan_specific_folders
        self.foldersToScanLabel.setEnabled(scan_specific)
        self.foldersToScan.setEnabled(scan_specific)
        self.addFolderToScan.setEnabled(scan_specific)
        self.removeFolderToScan.setEnabled(
            scan_specific and self.foldersToScan.count() > 1
        )

    def setIgnorePathWidgetValues(self) -> None:
        self.ignoredPaths.clear()
        if self.prefs.list_not_empty("ignored_paths"):
            self.ignoredPaths.addItems(self.prefs.ignored_paths)
            self.ignoredPaths.setCurrentRow(0)
        self.removePath.setEnabled(self.ignoredPaths.count())
        self.removeAllPath.setEnabled(self.ignoredPaths.count())
        self.ignoredPathsRe.setChecked(self.prefs.use_re_ignored_paths)

    def setAutomationWidgetValues(self) -> None:
        self.autoMount.setChecked(self.prefs.auto_mount)
        self.autoDownloadStartup.setChecked(self.prefs.auto_download_at_startup)
        self.autoDownloadInsertion.setChecked(
            self.prefs.auto_download_upon_device_insertion
        )
        self.autoEject.setChecked(self.prefs.auto_unmount)
        self.autoExit.setChecked(self.prefs.auto_exit)
        self.setAutoExitErrorState()

    def setAutoExitErrorState(self) -> None:
        if self.prefs.auto_exit:
            self.autoExitError.setChecked(self.prefs.auto_exit_force)
            self.autoExitError.setEnabled(True)
        else:
            self.autoExitError.setChecked(False)
            self.autoExitError.setEnabled(False)

    def setPerformanceValues(self, check_boxes_only: bool = False) -> None:
        self.generateThumbnails.setChecked(self.prefs.generate_thumbnails)
        self.useThumbnailCache.setChecked(
            self.prefs.use_thumbnail_cache and self.prefs.generate_thumbnails
        )
        self.fdoThumbnails.setChecked(
            self.prefs.save_fdo_thumbnails and self.prefs.generate_thumbnails
        )

        if not check_boxes_only:
            available = available_cpu_count(physical_only=True)
            self.maxCores.addItems(str(i + 1) for i in range(0, available))
            self.maxCores.setCurrentText(str(self.prefs.max_cpu_cores))

    def setPerfomanceEnabled(self) -> None:
        enable = self.prefs.generate_thumbnails
        self.useThumbnailCache.setEnabled(enable)
        self.fdoThumbnails.setEnabled(enable)
        self.maxCores.setEnabled(enable)
        self.coresLabel.setEnabled(enable)

    def setCacheValues(self) -> None:
        self.thumbnailNumber.setText(thousands(self.thumbnail_cache.no_thumbnails()))
        self.thumbnailSqlSize.setText(
            format_size_for_user(self.thumbnail_cache.db_size())
        )
        self.thumbnailCacheDaysKeep.setValue(self.prefs.keep_thumbnails_days)

    @pyqtSlot("PyQt_PyObject")
    def setCacheSize(self, size: int) -> None:
        self.thumbnailCacheSize.setText(format_size_for_user(size))

    def setTimeZoneValues(self) -> None:
        ignore = self.prefs.ignore_time_zone_changes
        self.ignoreTimeZone.setChecked(ignore)
        self.timeZoneOffsetResolution.setCurrentText(
            str(self.prefs.time_zone_offset_resolution)
        )
        self.timeZoneOffset.setEnabled(ignore)
        self.timeZoneOffsetLabel.setEnabled(ignore)

    def setErrorHandingValues(self) -> None:
        if self.prefs.conflict_resolution == int(ConflictResolution.skip):
            self.skipDownload.setChecked(True)
        else:
            self.addIdentifier.setChecked(True)
        if self.prefs.backup_duplicate_overwrite:
            self.overwriteBackup.setChecked(True)
        else:
            self.skipBackup.setChecked(True)

    def setWarningValues(self) -> None:
        self.warnDownloadingAll.setChecked(self.prefs.warn_downloading_all)
        if self.prefs.backup_files:
            self.warnBackupProblem.setChecked(self.prefs.warn_backup_problem)
        else:
            self.warnBackupProblem.setChecked(False)
        self.warnMissingLibraries.setChecked(
            self.prefs.warn_broken_or_missing_libraries
        )
        self.warnMetadata.setChecked(self.prefs.warn_fs_metadata_error)
        self.warnUnhandledFiles.setChecked(self.prefs.warn_unhandled_files)
        self.setAddExceptFilesValues()

        self.setBackupWarningEnabled()
        self.setUnhandledWarningEnabled()

    def setAddExceptFilesValues(self) -> None:
        self.exceptTheseFiles.clear()
        if self.prefs.list_not_empty("ignore_unhandled_file_exts"):
            self.exceptTheseFiles.addItems(self.prefs.ignore_unhandled_file_exts)
            self.exceptTheseFiles.setCurrentRow(0)

    def setBackupWarningEnabled(self) -> None:
        self.warnBackupProblem.setEnabled(self.prefs.backup_files)

    def setUnhandledWarningEnabled(self) -> None:
        enabled = self.prefs.warn_unhandled_files
        for widget in (
            self.exceptTheseFilesLabel,
            self.exceptTheseFiles,
            self.addExceptFiles,
        ):
            widget.setEnabled(enabled)
        count = bool(self.exceptTheseFiles.count())
        for widget in (self.removeExceptFiles, self.removeAllExceptFiles):
            widget.setEnabled(enabled and count)

    def setConsolidatedValues(self) -> None:
        enabled = self.prefs.consolidate_identical
        self.consolidateIdentical.setChecked(enabled)

        self.setTreatRawJpeg()
        self.setMarkRawJpeg()

        if enabled:
            # Must turn off the exclusive button group feature, or else
            # it's impossible to set all the radio buttons to False
            self.noConsolidationGroup.setExclusive(False)
            for widget in (
                self.clearCompletedDownloads,
                self.keepCompletedDownloads,
                self.promptCompletedDownloads,
            ):
                widget.setChecked(False)
            # Now turn it back on again
            self.noConsolidationGroup.setExclusive(True)
        else:
            self.setCompletedDownloadsValues()

        self.setConsolidatedEnabled()

    def setTreatRawJpeg(self) -> None:
        if self.prefs.consolidate_identical:
            if self.prefs.treat_raw_jpeg == int(TreatRawJpeg.one_photo):
                self.oneRawJpeg.setChecked(True)
            else:
                self.twoRawJpeg.setChecked(True)
        else:
            # Must turn off the exclusive button group feature, or else
            # it's impossible to set all the radio buttons to False
            self.treatRawJpegGroup.setExclusive(False)
            self.oneRawJpeg.setChecked(False)
            self.twoRawJpeg.setChecked(False)
            # Now turn it back on again
            self.treatRawJpegGroup.setExclusive(True)

    def setMarkRawJpeg(self) -> None:
        if self.prefs.consolidate_identical and self.twoRawJpeg.isChecked():
            v = self.prefs.mark_raw_jpeg
            if v == int(MarkRawJpeg.no_jpeg):
                self.noJpegWhenRaw.setChecked(True)
            elif v == int(MarkRawJpeg.no_raw):
                self.noRawWhenJpeg.setChecked(True)
            else:
                self.markRawJpeg.setChecked(True)
        else:
            # Must turn off the exclusive button group feature, or else
            # it's impossible to set all the radio buttons to False
            self.markRawJpegGroup.setExclusive(False)
            for widget in (self.noJpegWhenRaw, self.noRawWhenJpeg, self.markRawJpeg):
                widget.setChecked(False)
            # Now turn it back on again
            self.markRawJpegGroup.setExclusive(True)

    def setConsolidatedEnabled(self) -> None:
        enabled = self.prefs.consolidate_identical

        for widget in self.treatRawJpegGroup.buttons():
            widget.setEnabled(enabled)
        self.treatRawJpegLabel.setEnabled(enabled)

        self.setMarkRawJpegEnabled()

        for widget in (
            self.noconsolidationLabel,
            self.clearCompletedDownloads,
            self.keepCompletedDownloads,
            self.promptCompletedDownloads,
        ):
            widget.setEnabled(not enabled)

    def setMarkRawJpegEnabled(self) -> None:
        mark_enabled = self.prefs.consolidate_identical and self.twoRawJpeg.isChecked()
        for widget in self.markRawJpegGroup.buttons():
            widget.setEnabled(mark_enabled)
        self.markRawJpegLabel.setEnabled(mark_enabled)

    def setVersionCheckValues(self) -> None:
        self.checkNewVersion.setChecked(self.prefs.check_for_new_versions)
        self.includeDevRelease.setChecked(
            self.prefs.include_development_release or self.is_prerelease
        )
        self.setVersionCheckEnabled()

    def setVersionCheckEnabled(self) -> None:
        self.includeDevRelease.setEnabled(
            not (self.is_prerelease or not self.prefs.check_for_new_versions)
        )

    def setMetdataValues(self) -> None:
        self.ignoreMdatatimeMtpDng.setChecked(self.prefs.ignore_mdatatime_for_mtp_dng)
        self.forceExiftool.setChecked(self.prefs.force_exiftool)
        self.forceExiftoolVideo.setChecked(self.prefs.force_exiftool_video)

    def setCompletedDownloadsValues(self) -> None:
        s = self.prefs.completed_downloads
        if s == int(CompletedDownloads.keep):
            self.keepCompletedDownloads.setChecked(True)
        elif s == int(CompletedDownloads.clear):
            self.clearCompletedDownloads.setChecked(True)
        else:
            self.promptCompletedDownloads.setChecked(True)

    @pyqtSlot(int)
    def onlyExternalChanged(self, state: int) -> None:
        self.prefs.only_external_mounts = state == Qt.Checked
        if self.sdrApp is not None:
            self.sdrApp.search_for_devices_again = True

    @pyqtSlot(int)
    def noDcimChanged(self, state: int) -> None:
        self.prefs.scan_specific_folders = state == Qt.Checked
        self.setFoldersToScanState()
        if self.sdrApp is not None:
            self.sdrApp.scan_non_cameras_again = True

    @pyqtSlot(int)
    def ignoredPathsReChanged(self, state: int) -> None:
        self.prefs.use_re_ignored_paths = state == Qt.Checked
        if self.sdrApp is not None:
            self.sdrApp.scan_all_again = True

    def _equalizeWidgetWidth(self, widget_list) -> None:
        max_width = round(max(widget.width() for widget in widget_list))
        for widget in widget_list:
            widget.setFixedWidth(max_width)

    def showEvent(self, e: QShowEvent):
        self.chooser.minimum_width = self.restoreButton.width()
        self._equalizeWidgetWidth(self.device_right_side_buttons)
        self._equalizeWidgetWidth(self.device_list_widgets)
        super().showEvent(e)

    @pyqtSlot(int)
    def rowChanged(self, row: int) -> None:
        self.panels.setCurrentIndex(row)
        self.restoreButton.setToolTip(
            ("Сбросить настройки %s на значения по умолчанию") % self.chooser_items[row]
        )

    @pyqtSlot()
    def removeDeviceClicked(self) -> None:
        row = self.knownDevices.currentRow()
        item: QListWidgetItem = self.knownDevices.takeItem(row)
        known_device_type = item.type()
        if known_device_type == KnownDeviceType.volume_whitelist:
            self.prefs.del_list_value("volume_whitelist", item.text())
        elif known_device_type == KnownDeviceType.volume_blacklist:
            self.prefs.del_list_value("volume_blacklist", item.text())
        else:
            assert known_device_type == KnownDeviceType.camera_blacklist
            self.prefs.del_list_value("camera_blacklist", item.text())

        self.removeDevice.setEnabled(self.knownDevices.count())
        self.removeAllDevice.setEnabled(self.knownDevices.count())

        if self.sdrApp is not None:
            self.sdrApp.search_for_devices_again = True

    @pyqtSlot()
    def removeAllDeviceClicked(self) -> None:
        self.knownDevices.clear()
        self.prefs.volume_whitelist = [""]
        self.prefs.volume_blacklist = [""]
        self.prefs.camera_blacklist = [""]
        self.removeDevice.setEnabled(False)
        self.removeAllDevice.setEnabled(False)

        if self.sdrApp is not None:
            self.sdrApp.search_for_devices_again = True

    @pyqtSlot()
    def removeFolderToScanClicked(self) -> None:
        row = self.foldersToScan.currentRow()
        if row >= 0 and self.foldersToScan.count() > 1:
            item = self.foldersToScan.takeItem(row)
            self.prefs.del_list_value("folders_to_scan", item.text())
            self.removeFolderToScan.setEnabled(self.foldersToScan.count() > 1)

            if self.sdrApp is not None:
                self.sdrApp.scan_all_again = True

    @pyqtSlot()
    def addFolderToScanClicked(self) -> None:
        dlg = FoldersToScanDialog(prefs=self.prefs, parent=self)
        if dlg.exec():
            self.setFoldersToScanWidgetValues()

            if self.sdrApp is not None:
                self.sdrApp.scan_all_again = True

    @pyqtSlot()
    def removePathClicked(self) -> None:
        row = self.ignoredPaths.currentRow()
        if row >= 0:
            item = self.ignoredPaths.takeItem(row)
            self.prefs.del_list_value("ignored_paths", item.text())
            self.removePath.setEnabled(self.ignoredPaths.count())
            self.removeAllPath.setEnabled(self.ignoredPaths.count())

            if self.sdrApp is not None:
                self.sdrApp.scan_all_again = True

    @pyqtSlot()
    def removeAllPathClicked(self) -> None:
        self.ignoredPaths.clear()
        self.prefs.ignored_paths = [""]
        self.removePath.setEnabled(False)
        self.removeAllPath.setEnabled(False)

        if self.sdrApp is not None:
            self.sdrApp.scan_all_again = True

    @pyqtSlot()
    def addPathClicked(self) -> None:
        dlg = IgnorePathDialog(prefs=self.prefs, parent=self)
        if dlg.exec():
            self.setIgnorePathWidgetValues()

            if self.sdrApp is not None:
                self.sdrApp.scan_all_again = True

    @pyqtSlot()
    def ignorePathsReLabelClicked(self) -> None:
        self.ignoredPathsRe.click()

    @pyqtSlot(int)
    def languagesChanged(self, index: int) -> None:
        if index == 0:
            self.prefs.language = ""
            logging.info("Resetting user interface language to system default")
        elif index > 0:
            self.prefs.language = self.languages.currentData()
            logging.info("Setting user interface language to %s", self.prefs.language)

    @pyqtSlot(int)
    def autoMountChanged(self, state: int) -> None:
        on = state == Qt.Checked
        self.prefs.auto_mount = on
        if self.sdrApp.use_udsisks:
            if not on:
                self.sdrApp.start_monitoring_mount_count = True
                self.sdrApp.stop_monitoring_mount_count = False
            else:
                self.sdrApp.stop_monitoring_mount_count = True
                self.sdrApp.start_monitoring_mount_count = False

    @pyqtSlot(int)
    def autoDownloadStartupChanged(self, state: int) -> None:
        self.prefs.auto_download_at_startup = state == Qt.Checked

    @pyqtSlot(int)
    def autoDownloadInsertionChanged(self, state: int) -> None:
        self.prefs.auto_download_upon_device_insertion = state == Qt.Checked

    @pyqtSlot(int)
    def autoEjectChanged(self, state: int) -> None:
        self.prefs.auto_unmount = state == Qt.Checked

    @pyqtSlot(int)
    def autoExitChanged(self, state: int) -> None:
        auto_exit = state == Qt.Checked
        self.prefs.auto_exit = auto_exit
        self.setAutoExitErrorState()
        if not auto_exit:
            self.prefs.auto_exit_force = False

    @pyqtSlot(int)
    def autoExitErrorChanged(self, state: int) -> None:
        self.prefs.auto_exit_force = state == Qt.Checked

    @pyqtSlot(int)
    def generateThumbnailsChanged(self, state: int) -> None:
        self.prefs.generate_thumbnails = state == Qt.Checked
        self.setPerformanceValues(check_boxes_only=True)
        self.setPerfomanceEnabled()

    @pyqtSlot(int)
    def useThumbnailCacheChanged(self, state: int) -> None:
        if self.prefs.generate_thumbnails:
            self.prefs.use_thumbnail_cache = state == Qt.Checked

    @pyqtSlot(int)
    def fdoThumbnailsChanged(self, state: int) -> None:
        if self.prefs.generate_thumbnails:
            self.prefs.save_fdo_thumbnails = state == Qt.Checked

    @pyqtSlot(int)
    def thumbnailCacheDaysKeepChanged(self, value: int) -> None:
        self.prefs.keep_thumbnails_days = value

    @pyqtSlot(int)
    def maxCoresChanged(self, index: int) -> None:
        if index >= 0:
            self.prefs.max_cpu_cores = int(self.maxCores.currentText())

    @pyqtSlot()
    def purgeCacheClicked(self) -> None:
        message = (
            "Вы хотите очистить кэш миниатюр? Кэш будет очищен при следующем запуске "
            "программы."
        )
        msgBox = standardMessageBox(
            parent=self,
            title=("Очистить кэш миниатюр"),
            message=message,
            standardButtons=QMessageBox.Yes | QMessageBox.No,
            rich_text=False,
        )

        if msgBox.exec_() == QMessageBox.Yes:
            self.prefs.purge_thumbnails = True
            self.prefs.optimize_thumbnail_db = False
        else:
            self.prefs.purge_thumbnails = False

    @pyqtSlot()
    def optimizeCacheClicked(self) -> None:
        message = (
            "Вы хотите оптимизировать кэш миниатюр? Кэш будет соптимизирован при "
            "следующем запуске программы."
        )
        msgBox = standardMessageBox(
            parent=self,
            title=("Оптимизировать кэш миниатюр"),
            message=message,
            standardButtons=QMessageBox.Yes | QMessageBox.No,
            rich_text=False,
        )
        if msgBox.exec_() == QMessageBox.Yes:
            self.prefs.purge_thumbnails = False
            self.prefs.optimize_thumbnail_db = True
        else:
            self.prefs.optimize_thumbnail_db = False

    @pyqtSlot(int)
    def ignoreTimeZoneChanged(self, state: int) -> None:
        ignore = state == Qt.Checked
        self.prefs.ignore_time_zone_changes = ignore
        self.timeZoneOffset.setEnabled(ignore)
        self.timeZoneOffsetLabel.setEnabled(ignore)

    @pyqtSlot(int)
    def timeZoneOffsetResolutionChanged(self, index: int) -> None:
        self.prefs.time_zone_offset_resolution = int(
            self.timeZoneOffsetResolution.currentText()
        )

    @pyqtSlot(QAbstractButton)
    def downloadErrorGroupClicked(self, button: QRadioButton) -> None:
        if self.downloadErrorGroup.checkedButton() == self.skipDownload:
            self.prefs.conflict_resolution = int(ConflictResolution.skip)
        else:
            self.prefs.conflict_resolution = int(ConflictResolution.add_identifier)

    @pyqtSlot(QAbstractButton)
    def backupErrorGroupClicked(self, button: QRadioButton) -> None:
        self.prefs.backup_duplicate_overwrite = (
            self.backupErrorGroup.checkedButton() == self.overwriteBackup
        )

    @pyqtSlot(int)
    def warnDownloadingAllChanged(self, state: int) -> None:
        self.prefs.warn_downloading_all = state == Qt.Checked

    @pyqtSlot(int)
    def warnBackupProblemChanged(self, state: int) -> None:
        self.prefs.warn_backup_problem = state == Qt.Checked

    @pyqtSlot(int)
    def warnMissingLibrariesChanged(self, state: int) -> None:
        self.prefs.warn_broken_or_missing_libraries = state == Qt.Checked

    @pyqtSlot(int)
    def warnMetadataChanged(self, state: int) -> None:
        self.prefs.warn_fs_metadata_error = state == Qt.Checked

    @pyqtSlot(int)
    def warnUnhandledFilesChanged(self, state: int) -> None:
        self.prefs.warn_unhandled_files = state == Qt.Checked
        self.setUnhandledWarningEnabled()

    @pyqtSlot()
    def addExceptFilesClicked(self) -> None:
        dlg = ExceptFileExtDialog(prefs=self.prefs, parent=self)
        if dlg.exec():
            self.setAddExceptFilesValues()

    @pyqtSlot()
    def removeExceptFilesClicked(self) -> None:
        row = self.exceptTheseFiles.currentRow()
        if row >= 0:
            item = self.exceptTheseFiles.takeItem(row)
            self.prefs.del_list_value("ignore_unhandled_file_exts", item.text())
            self.removeExceptFiles.setEnabled(self.exceptTheseFiles.count())
            self.removeAllExceptFiles.setEnabled(self.exceptTheseFiles.count())

    @pyqtSlot()
    def removeAllExceptFilesClicked(self) -> None:
        self.exceptTheseFiles.clear()
        self.prefs.ignore_unhandled_file_exts = [""]
        self.removeExceptFiles.setEnabled(False)
        self.removeAllExceptFiles.setEnabled(False)

    @pyqtSlot(int)
    def consolidateIdenticalChanged(self, state: int) -> None:
        self.prefs.consolidate_identical = state == Qt.Checked
        self.setConsolidatedValues()
        self.setConsolidatedEnabled()

    @pyqtSlot(QAbstractButton)
    def treatRawJpegGroupClicked(self, button: QRadioButton) -> None:
        if button == self.oneRawJpeg:
            self.prefs.treat_raw_jpeg = int(TreatRawJpeg.one_photo)
        else:
            self.prefs.treat_raw_jpeg = int(TreatRawJpeg.two_photos)
        self.setMarkRawJpeg()
        self.setMarkRawJpegEnabled()

    @pyqtSlot(QAbstractButton)
    def markRawJpegGroupClicked(self, button: QRadioButton) -> None:
        if button == self.noJpegWhenRaw:
            self.prefs.mark_raw_jpeg = int(MarkRawJpeg.no_jpeg)
        elif button == self.noRawWhenJpeg:
            self.prefs.mark_raw_jpeg = int(MarkRawJpeg.no_raw)
        else:
            self.prefs.mark_raw_jpeg = int(MarkRawJpeg.both)

    @pyqtSlot(int)
    def noJpegWhenRawChanged(self, state: int) -> None:
        self.prefs.do_not_mark_jpeg = state == Qt.Checked

    @pyqtSlot(int)
    def noRawWhenJpegChanged(self, state: int) -> None:
        self.prefs.do_not_mark_raw = state == Qt.Checked

    @pyqtSlot(int)
    def checkNewVersionChanged(self, state: int) -> None:
        do_check = state == Qt.Checked
        self.prefs.check_for_new_versions = do_check
        self.setVersionCheckEnabled()

    @pyqtSlot(int)
    def includeDevReleaseChanged(self, state: int) -> None:
        self.prefs.include_development_release = state == Qt.Checked

    @pyqtSlot(int)
    def ignoreMdatatimeMtpDngChanged(self, state: int) -> None:
        self.prefs.ignore_mdatatime_for_mtp_dng = state == Qt.Checked

    @pyqtSlot(int)
    def forceExiftoolChanged(self, state: int) -> None:
        self.prefs.force_exiftool = state == Qt.Checked

    @pyqtSlot(int)
    def forceExiftoolVideoChanged(self, state: int) -> None:
        self.prefs.force_exiftool_video = state == Qt.Checked

    @pyqtSlot(QAbstractButton)
    def noConsolidationGroupClicked(self, button: QRadioButton) -> None:
        if button == self.keepCompletedDownloads:
            self.prefs.completed_downloads = int(CompletedDownloads.keep)
        elif button == self.clearCompletedDownloads:
            self.prefs.completed_downloads = int(CompletedDownloads.clear)
        else:
            self.prefs.completed_downloads = int(CompletedDownloads.prompt)

    @pyqtSlot()
    def restoreDefaultsClicked(self) -> None:
        row = self.chooser.currentRow()
        if row == 0:
            for value in (
                "only_external_mounts",
                "scan_specific_folders",
                "folders_to_scan",
                "ignored_paths",
                "use_re_ignored_paths",
            ):
                self.prefs.restore(value)
            self.removeAllDeviceClicked()
            self.setDeviceWidgetValues()
        elif row == 1:
            self.prefs.restore("language")
            self.languages.setCurrentIndex(0)
        elif row == 2:
            for value in (
                "auto_mount",
                "auto_download_at_startup",
                "auto_download_upon_device_insertion",
                "auto_unmount",
                "auto_exit",
                "auto_exit_force",
            ):
                self.prefs.restore(value)
            self.setAutomationWidgetValues()
        elif row == 3:
            for value in (
                "generate_thumbnails",
                "use_thumbnail_cache",
                "save_fdo_thumbnails",
                "max_cpu_cores",
                "keep_thumbnails_days",
            ):
                self.prefs.restore(value)
            self.setPerformanceValues(check_boxes_only=True)
            self.maxCores.setCurrentText(str(self.prefs.max_cpu_cores))
            self.setPerfomanceEnabled()
            self.thumbnailCacheDaysKeep.setValue(self.prefs.keep_thumbnails_days)
        elif row == 4:
            for value in ("ignore_time_zone_changes", "time_zone_offset_resolution"):
                self.prefs.restore(value)
                self.setTimeZoneValues()
        elif row == 5:
            for value in ("conflict_resolution", "backup_duplicate_overwrite"):
                self.prefs.restore(value)
            self.setErrorHandingValues()
        elif row == 6:
            for value in (
                "warn_downloading_all",
                "warn_backup_problem",
                "warn_broken_or_missing_libraries",
                "warn_fs_metadata_error",
                "warn_unhandled_files",
                "ignore_unhandled_file_exts",
            ):
                self.prefs.restore(value)
            self.setWarningValues()
        elif row == 7 and CONSOLIDATION_IMPLEMENTED:
            for value in (
                "completed_downloads",
                "consolidate_identical",
                "one_raw_jpeg",
                "do_not_mark_jpeg",
                "do_not_mark_raw",
            ):
                self.prefs.restore(value)
            self.setConsolidatedValues()
        elif (row == 8 and CONSOLIDATION_IMPLEMENTED) or (
            row == 7 and not CONSOLIDATION_IMPLEMENTED
        ):
            for value in (
                "include_development_release",
                "ignore_mdatatime_for_mtp_dng",
                "force_exiftool",
                "force_exiftool_video",
            ):
                self.prefs.restore(value)
            if not CONSOLIDATION_IMPLEMENTED:
                self.prefs.restore("completed_downloads")
            self.setMetdataValues()
            if not CONSOLIDATION_IMPLEMENTED:
                self.setCompletedDownloadsValues()

    @pyqtSlot()
    def helpButtonClicked(self) -> None:
        row = self.chooser.currentRow()
        if row == 0:
            location = "#devicepreferences"
        elif row == 1:
            location = "#languagepreferences"
        elif row == 2:
            location = "#automationpreferences"
        elif row == 3:
            location = "#thumbnailpreferences"
        elif row == 4:
            location = "#timezonehandling"
        elif row == 5:
            location = "#errorhandlingpreferences"
        elif row == 6:
            location = "#warningpreferences"
        elif row == 7:
            if CONSOLIDATION_IMPLEMENTED:
                location = "#consolidationpreferences"
            else:
                location = "#miscellaneousnpreferences"
        elif row == 8:
            location = "#miscellaneousnpreferences"
        else:
            location = ""

        webbrowser.open_new_tab(
            f"https://google.com/{location}"
        )

    def closeEvent(self, event: QCloseEvent) -> None:
        self.cacheSizeThread.quit()
        self.cacheSizeThread.wait(1000)
        event.accept()


class PreferenceAddDialog(QDialog):
    """
    Base class for adding value to pref list
    """

    def __init__(
        self,
        prefs: Preferences,
        title: str,
        instruction: str,
        label: str,
        pref_value: str,
        parent=None,
    ) -> None:
        super().__init__(parent=parent)

        self.prefs = prefs
        self.pref_value = pref_value

        self.setWindowTitle(title)

        self.instructionLabel = QLabel(instruction)
        self.instructionLabel.setWordWrap(False)
        layout = QVBoxLayout()
        self.setLayout(layout)

        self.valueEdit = QLineEdit()
        formLayout = QFormLayout()
        formLayout.addRow(label, self.valueEdit)

        buttons = QDialogButtonBox(QDialogButtonBox.Cancel | QDialogButtonBox.Ok)
        translateDialogBoxButtons(buttons)
        buttons.rejected.connect(self.reject)
        buttons.accepted.connect(self.accept)

        layout.addWidget(self.instructionLabel)
        layout.addLayout(formLayout)
        layout.addWidget(buttons)

    def accept(self):
        value = self.valueEdit.text()
        if value:
            self.prefs.add_list_value(self.pref_value, value)
        super().accept()


class FoldersToScanDialog(PreferenceAddDialog):
    """
    Dialog prompting for a folder on devices to scan for photos and videos
    """

    def __init__(self, prefs: Preferences, parent=None) -> None:
        super().__init__(
            prefs=prefs,
            title=("Укажите папку для сканирования"),
            instruction=(
                "Укажите папку для сканирования на наличие фото и видео"
            ),
            label=("Папка:"),
            pref_value="folders_to_scan",
            parent=parent,
        )


class IgnorePathDialog(PreferenceAddDialog):
    """
    Dialog prompting for a path to ignore when scanning devices
    """

    def __init__(self, prefs: Preferences, parent=None) -> None:
        super().__init__(
            prefs=prefs,
            title=("Введите путь, который хотите игнорировать"),
            instruction=(
                "Задайте путь, который никогда не будет проверяться на наличие фото и видео"
            ),
            label=("Путь:"),
            pref_value="ignored_paths",
            parent=parent,
        )


class ExceptFileExtDialog(PreferenceAddDialog):
    """
    Dialog prompting for file extensions never to warn about
    """

    def __init__(self, prefs: Preferences, parent=None) -> None:
        super().__init__(
            prefs=prefs,
            title=("Введите расширение файла"),
            instruction=("Задайте расширение файла (без предшествующей точки)"),
            label=("Расширение:"),
            pref_value="ignore_unhandled_file_exts",
            parent=parent,
        )

    def exts(self, exts: list[str]) -> str:
        return make_internationalized_list([ext.upper() for ext in exts])

    def accept(self):
        value = self.valueEdit.text()
        if value:
            while value.startswith("."):
                value = value[1:]
            value = value.upper()
            if value.lower() in ALL_KNOWN_EXTENSIONS:
                title = ("Неверное расширение файла")
                message = (
                    (
                        "Расширение файлов <b>%s</b> известно SmartVision Guard, поэтому нет "
                        "необходимости предупреждать об их наличии."
                    )
                    % value
                )
                details = (
                    "Известные типы файлов:\n\n"
                    "Фото:\n%(photos)s\n\nВидео:\n%(videos)s\n\n"
                    "Аудио:\n%(audio)s\n\nДругие:\n%(other)s"
                ) % dict(
                    photos=self.exts(PHOTO_EXTENSIONS),
                    videos=self.exts(VIDEO_EXTENSIONS + VIDEO_THUMBNAIL_EXTENSIONS),
                    audio=self.exts(AUDIO_EXTENSIONS),
                    other=self.exts(["xmp"]),
                )
                msgBox = standardMessageBox(
                    parent=self,
                    title=title,
                    message=message,
                    rich_text=True,
                    standardButtons=QMessageBox.Ok,
                    iconType=QMessageBox.Information,
                )
                msgBox.setDetailedText(details)
                msgBox.exec()
                self.valueEdit.setText(value)
                self.valueEdit.selectAll()
                return
            else:
                self.prefs.add_list_value(self.pref_value, value)
        QDialog.accept(self)


class CacheSize(QObject):
    size = pyqtSignal("PyQt_PyObject")  # don't convert python int to C++ int

    @pyqtSlot()
    def start(self) -> None:
        self.thumbnail_cache = ThumbnailCacheSql(create_table_if_not_exists=False)

    @pyqtSlot()
    def getCacheSize(self) -> None:
        self.size.emit(self.thumbnail_cache.cache_size())


if __name__ == "__main__":
    # Application development test code:

    app = QApplication([])

    app.setOrganizationName("SmartVision Guard")
    app.setOrganizationDomain("sdr-technology.ru")
    app.setApplicationName("SmartVision Guard")

    prefs = Preferences()

    prefDialog = PreferencesDialog(prefs)
    prefDialog.show()
    app.exec_()
