#
#

import io
import logging
import os
import traceback

from PyQt5.QtWidgets import QApplication, QMessageBox

from svg.internationalisation.install import install_gettext
from svg.iplogging import full_log_file_path
from svg.prefs.preferences import Preferences
from svg.storage.storage import get_uri
from svg.tools.utilities import bug_report_full_tar_path, create_bugreport_tar
from svg.ui.viewutils import standardMessageBox

install_gettext()

message_box_displayed = False
exceptions_notified = set()
please_report_problem_body = (
    'Пожалуйста, сообщите о проблеме в <a href=\"{website}\">{website}</a>.'
    "<br><br>В вашем отчете об ошибке опишите что вы ожидали и что произошло."
    "<br><br>Отчет об ошибке также должен содержать настройки программы и файлы "
    "журналов. Для создания файла с дополнительной информацией нажмите Сохранить."
)

tar_created_title = ("Дополнительная информация Сохранена")
tar_created_body = (
    "Дополнительная информация для отчета об ошибке создана в tar-файле в "
    "домашней папке: <pre>{tarfile}</pre>Вам необходимо собственноручно "
    "прикрепить этот файл к отчету об ошибке. Он не будет прикреплен "
    'автоматически.<br><br>Нажмите <a href=\"{uri}\">здесь</a> для просмотра файла '
    "в вашем Диспетчере файлов."
)
tar_error_title = ("Ошибка при создании Дополнительной Информации")
tar_error_header = (
    "Дополнительная информация для отчета об ошибке не была создана. Все равно "
    "сообщите об ошибке, пожалуйста."
)
tar_error_body = (
    "Включите в ваш отчет об ошибке файлы журнала программы. Отчет об ошибке "
    "должен содержать <i>{log_file}</i>, но включение других файлов журналов "
    "часто бывает полезно.<br><br>Если возможно, пожалуйста также добавьте файл "
    'настроек программы <i>{config_file}</i>.<br><br>Нажмите <a href=\"{log_path}'
    '\">здесь</a> для открытия папки с журналами, и <a href=\"{config_path}'
    '\">здесь</a> для открытия папки настроек.'
)
upgrade_message = (
    'Обновление до <a href=\"{website}\">latest version</a> позволит вам '
    "определить была ли устранена замеченая вами проблема."
)


def save_bug_report_tar(config_file: str, full_log_file_path: str) -> None:
    """
    Save a tar file in the user's home directory with logging files and config file.
    Inform the user of the result using QMessageBox.

    :param config_file: full path to the config file
    :param full_log_file_path: full path to the directory with the log files
    """

    bug_report_full_tar = bug_report_full_tar_path()

    logging.info("Creating bug report tar file %s", bug_report_full_tar)
    log_path, log_file = os.path.split(full_log_file_path)
    if create_bugreport_tar(
        full_tar_name=bug_report_full_tar,
        log_path=log_path,
        full_config_file=config_file,
    ):
        body = tar_created_body.format(
            tarfile=os.path.split(bug_report_full_tar)[1],
            uri=get_uri(full_file_name=bug_report_full_tar),
        )
        messagebox = standardMessageBox(
            message=body,
            rich_text=True,
            title=tar_created_title,
            standardButtons=QMessageBox.Ok,
        )
        messagebox.exec_()
    else:
        # There was some kind of problem generating the tar file, e.g. no free space
        log_uri = get_uri(log_path)
        config_path, config_file = os.path.split(config_file)
        config_uri = get_uri(path=config_path)

        body = tar_error_body.format(
            log_path=log_uri,
            log_file=log_file,
            config_path=config_uri,
            config_file=config_file,
        )
        message = f"<b>{tar_error_header}</b><br><br>{body}"
        messageBox = standardMessageBox(
            message=message,
            rich_text=True,
            title=tar_error_title,
            standardButtons=QMessageBox.Ok,
        )
        messageBox.exec_()


def excepthook(exception_type, exception_value, traceback_object) -> None:
    """
    Global function to catch unhandled exceptions.

    Inspired by function of the same name in the Eric project, but subsequently heavily
    modified.
    """

    if traceback_object is not None:
        frame = traceback_object.tb_frame
        filename = frame.f_code.co_filename
        lineno = traceback_object.tb_lineno
    else:
        lineno = -1
        filename = "unknown"
    key = f"{filename}{lineno}"

    global message_box_displayed

    tb_file = io.StringIO()
    traceback.print_exception(
        exception_type, exception_value, traceback_object, limit=None, file=tb_file
    )
    tb_file.seek(0)
    traceback_info = tb_file.read()

    logging.error("An unhandled exception occurred")
    logging.error(traceback_info)

    if not message_box_displayed and key not in exceptions_notified:
        message_box_displayed = True
        exceptions_notified.add(key)
        prefs = Preferences()

        title = ("Проблема в SmartVision Guard")

        if QApplication.instance():
            header = ("В SmartVision Guard обнаружена проблема")

            only_notification = (
                "Если подобная проблема возникнет снова до выхода из программы, больше "
                "уведомлений о ней не будет."
            )

            body = please_report_problem_body.format(
                website="https://sdr-technology.ru"
            )

            message = f"<b>{header}</b><br><br>{body}<br><br>{only_notification}"

            errorbox = standardMessageBox(
                message=message,
                rich_text=True,
                title=title,
                standardButtons=QMessageBox.Save | QMessageBox.Cancel,
                defaultButton=QMessageBox.Save,
            )
            errorbox.setDetailedText(traceback_info)
            if errorbox.exec_() == QMessageBox.Save:
                save_bug_report_tar(
                    config_file=prefs.settings_path(),
                    full_log_file_path=full_log_file_path(),
                )
        message_box_displayed = False
