#
#

"""
Commandline argument parser for SmartVision Guard
"""

import builtins
import platform
from argparse import ArgumentParser, RawDescriptionHelpFormatter
from pathlib import Path

try:
    from svg import __about__ as __about__
    from svg.internationalisation.install import install_gettext
    from svg.internationalisation.utilities import make_internationalized_list
    from svg.metadata.fileextensions import OTHER_PHOTO_EXTENSIONS

    install_gettext()
except ImportError:
    # The script is being run at build time
    # Module imports are unavailable

    def no_translation_performed(s: str) -> str:
        return s

    builtins.__dict__["_"] = no_translation_performed

    here = Path(__file__).parent
    with open(here / "__about__.py") as f:
        about = {}
        exec(f.read(), about)

    # Convert about dictionary to class
    class About:
        pass
    __about__ = About()
    __about__.__dict__.update(about)

    with open(here / "metadata/fileextensions.py") as f:
        file_extensions = {}
        exec(f.read(), file_extensions)
        OTHER_PHOTO_EXTENSIONS = file_extensions["OTHER_PHOTO_EXTENSIONS"]
    with open(here / "internationalisation/utilities.py") as f:
        utilities = {}
        exec(f.read(), utilities)
        make_internationalized_list = utilities["make_internationalized_list"]


def get_parser(formatter_class=RawDescriptionHelpFormatter) -> ArgumentParser:
    parser = ArgumentParser(
        prog=__about__.__title__,
        description=__about__.__summary__,
        epilog=__about__.__help_epilog__,
        formatter_class=formatter_class,
    )

    parser.add_argument(
        "--version",
        action="version",
        version=f"%(prog)s {__about__.__version__}",
    )
    parser.add_argument(
        "--detailed-version",
        action="store_true",
        help=("Показать версию программы и ее библиотек и выйти."),
    )
    parser.add_argument(
        "-v",
        "--verbose",
        action="store_true",
        dest="verbose",
        help=("Показывать информацию программы при запуске из командной строки."),
    )
    parser.add_argument(
        "--debug",
        action="store_true",
        dest="debug",
        help=("Показывать отладочную информацию при запуске из командной строки."),
    )
    parser.add_argument(
        "-e",
        "--extensions",
        action="store_true",
        dest="extensions",
        help=("Показать расширения фото и видео, известные программе и завершить."),
    )
    parser.add_argument(
        "--photo-renaming",
        choices=["on", "off"],
        dest="photo_renaming",
        help=("Включить или выключить переименование фото."),
    )
    parser.add_argument(
        "--video-renaming",
        choices=["on", "off"],
        dest="video_renaming",
        help=("Включить или выключить переименование видео."),
    )
    parser.add_argument(
        "-a",
        "--auto-detect",
        choices=["on", "off"],
        dest="auto_detect",
        help=(
            "Включить или выключить автоматическое определение устройств, с которых осуществлять загрузку."
        ),
    )
    parser.add_argument(
        "-t",
        "--this-computer",
        choices=["on", "off"],
        dest="this_computer_source",
        help=("Включить или выключить загрузку с этого компьютера."),
    )
    parser.add_argument(
        "--this-computer-location",
        type=str,
        metavar=("ПУТЬ"),
        dest="this_computer_location",
        help=("ПУТЬ на этом компьютере из компьютере, из которого осуществлять загрузку."),
    )
    parser.add_argument(
        "--photo-destination",
        type=str,
        metavar=("ПУТЬ"),
        dest="photo_location",
        help=("ПУТЬ куда будут загружаться фото."),
    )
    parser.add_argument(
        "--video-destination",
        type=str,
        metavar=("ПУТЬ"),
        dest="video_location",
        help=("ПУТЬ куда будут загружаться видео."),
    )
    parser.add_argument(
        "-b",
        "--backup",
        choices=["on", "off"],
        dest="backup",
        help=("Включение или выключение создания резервных копий фото и видео во время загрузки."),
    )
    parser.add_argument(
        "--backup-auto-detect",
        choices=["on", "off"],
        dest="backup_auto_detect",
        help=("Включение или выключение автоматического определения устройств для хранения резервных копий."),
    )
    parser.add_argument(
        "--photo-backup-identifier",
        type=str,
        metavar=("ПАПКА"),
        dest="photo_backup_identifier",
        help=(
            "ПАПКА в которой хранятся резервные копии на автоматически определяемом "
            "устройстве для хранения резервных копий фото, имя этой папки используется "
            "для определения, используется ли это устройство для хранения резервных "
            "копий. Создайте папку с таким именем на каждом устройстве, которое вы хотите "
            "использовать для хранения резервных копий фото."
       ),
    )
    parser.add_argument(
        "--video-backup-identifier",
        type=str,
        metavar=("ПАПКА"),
        dest="video_backup_identifier",
        help=(
            "ПАПКА в которой хранятся резервные копии на автоматически определяемом "
            "устройстве для хранения резервных копий видео, имя этой папки используется "
            "для определения, используется ли это устройство для хранения резервных "
            "копий. Создайте папку с таким именем на каждом устройстве, которое вы хотите "
            "использовать для хранения резервных копий видео."
        ),
    )
    parser.add_argument(
        "--photo-backup-location",
        type=str,
        metavar=("ПУТЬ"),
        dest="photo_backup_location",
        help=(
            "ПУТЬ, где будут храниться резервные копии фото в случае, когда "
            "автоматическое определение устройств для хранения резервных копий отключено."
        ),
    )
    parser.add_argument(
        "--video-backup-location",
        type=str,
        metavar=("ПУТЬ"),
        dest="video_backup_location",
        help=(
            "ПУТЬ, где будут храниться резервные копии видео в случае, когда "
            "автоматическое определение устройств для хранения резервных копий отключено."
        ),
    )
    parser.add_argument(
        "--ignore-other-photo-file-types",
        action="store_true",
        dest="ignore_other",
        help=("Игнорировать фото со следующими расширениями: %s")
        % make_internationalized_list([s.upper() for s in OTHER_PHOTO_EXTENSIONS]),
    )
    parser.add_argument(
        "--auto-download-startup",
        dest="auto_download_startup",
        choices=["on", "off"],
        help=(
            "Включение или выключение запуска загрузки сразу после старта программы."
        ),
    )
    parser.add_argument(
        "--auto-download-device-insertion",
        dest="auto_download_insertion",
        choices=["on", "off"],
        help=("Включение или выключение запуска загрузки сразу после подключения устройства."),
    )
    parser.add_argument(
        "--thumbnail-cache",
        dest="thumb_cache",
        choices=["on", "off"],
        help=(
            "Включение или выключение использования кэша миниатюр SmartVision Guard. "
            "Выключение не приводит к удалению существующего содержимого кэша."
        ),
    )
    parser.add_argument(
        "--delete-thumbnail-cache",
        dest="delete_thumb_cache",
        action="store_true",
        help=(
            "Удалить все миниатюры в кэше миниатюр SmartVision Guard и выйти."
        ),
    )
    parser.add_argument(
        "--forget-remembered-files",
        dest="forget_files",
        action="store_true",
        help=("Забыть файлы, которые уже были загружены и выйти."),
    )
    parser.add_argument(
        "--reset",
        action="store_true",
        dest="reset",
        help=(
            "Сбросить все настройки программы на значения по умолчанию, удалить все "
            "миниатюры в кэше миниатюр, забыть, какие файлы уже были загружены и выйти."
        ),
    )
    parser.add_argument(
        "--log-gphoto2",
        action="store_true",
        help=("Включать отладочную информацию gphoto2 в лог файлы."),
    )

    parser.add_argument(
        "--camera-info",
        action="store_true",
        help=("Вывести в терминал информацию о подключенных камерах и выйти."),
    )

    parser.add_argument(
        "--force-system-theme",
        action="store_true",
        default=False,
        help=("Использовать системную тему Qt вместо встроеной темы"),
    )

    parser.add_argument(
        "path",
        nargs="?",
        help=(
            "Необязательное значение, которое при указании анализируется для определения,"
            "представляет ли оно автоматически обнаруженное устройство или путь на этом "
            "компьютере. Если ПУТЬ представляет автоматически обнаруженное устройство, "
            "автоматическое обнаружение устройств включается, как в параметре '--auto-detect'."
            "Кроме того, загрузка с указанного вручную пути, как в параметре '--this-computer-location',"
            "отключается. В противном случае предполагается, что ПУТЬ — это указанный вручную"
            "путь, как в параметре '--this-computer-location', и в этом случае загрузка с этого"
            "компьютера включена, а загрузка с автоматически обнаруженных устройств отключается."           
        ),
    )

    if platform.system() == "Linux":
        parser.add_argument(
            "-platform",
            type=str,
            choices=["wayland", "xcb"],
            help=("Запустить программу в wayland или обычном X11"),
        )

    return parser
